<?php

namespace App\sys\Services\Accounting;

use App\sys\Repository\Accounting\TreeAccountingTranslationRepository;
use App\sys\Services;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;

class TreeAccountingTranslationServices extends Services
{
    private TreeAccountingTranslationRepository $repo;

    public function __construct()
    {
        $this->repo = new TreeAccountingTranslationRepository;
    }

    public function save(array $rows, int $treeId)
    {
        $rules = ['tree_id' => 'required|integer|exists:tree_accounting,id'];
        $combinations = [];
        $customErrors = [];
        foreach ($rows as $index => $row) {
            $rowId = $row['id'] ?? null;
            $languageId = $row['language_id'] ?? null;
            $rules["$index.id"] = 'sometimes|nullable|exists:tree_accounting_translation,id';
            $rules["$index.language_id"] = [
                Rule::requiredIf(empty($rowId)),
                'exists:languages,id',
                Rule::unique('tree_accounting_translation', 'lang_id')
                    ->where(fn ($q) => $q->where('tree_accounting_id', $treeId))
                    ->ignore($rowId),
            ];
            $rules["$index.title"] = 'nullable|string|max:250';
            $rules["$index.description"] = 'nullable|string';
            if ($languageId) {
                $key = $treeId.'-'.$languageId;
                if (in_array($key, $combinations)) {
                    $customErrors["$index.language_id"] = 'Same language duplicated for the same item.';
                } else {
                    $combinations[] = $key;
                }
            }
        }

        $validator = Validator::make(array_merge($rows, ['tree_id' => $treeId]), $rules);
        $validator->after(function ($validator) use ($customErrors) {
            foreach ($customErrors as $field => $message) {
                $validator->errors()->add($field, $message);
            }
        });
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->repo->save($rows, $treeId);
    }

    public function get(int $treeId)
    {
        return $this->repo->get($treeId);
    }

    public function del(int $id)
    {
        $rules = [
            'id' => 'required|integer|exists:tree_accounting_translation,id',
        ];
        $validator = Validator::make(['id' => $id], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->repo->del($id);
    }
}
