<?php

namespace App\sys\Services\General;

use App\Models\General\Currency;
use App\sys\Enums\CompanyAccountMapping;
use App\sys\Helper;
use App\sys\Repository\Accounting\TreeAccountingRepository;
use App\sys\Repository\General\CompanyRepository;
use App\sys\Services;
use Illuminate\Support\Facades\Validator;

class CompanyService extends Services
{
    protected CompanyRepository $companyRepository;

    public function __construct(CompanyRepository $companyRepository)
    {
        $this->companyRepository = $companyRepository;
    }

    public function getPaginatedCompanies()
    {
        return $this->companyRepository->getPaginated();
    }

    public function getCompanyById(int $id)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:companies,id'],
        ];
        $validator = Validator::make(['id' => $id], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->companyRepository->findByIdOrFail($id);
    }

    public function getActive()
    {
        return $this->companyRepository->getActive();
    }

    public function createCompany($request)
    {
        // TODO: Fix this to accept 2 images
        if (! $this->validateCompany($request)) {
            return false;
        }
        $saveLogo = Helper::saveFiles($request['invoice_logo'], 'upload/company', 'image');
        if (! $saveLogo['status']) {
            $this->setError(['invoice_logo' => [$saveLogo['errors']]]);

            return false;
        }
        $saveLogo = Helper::saveFiles($request['invoice_logo'], 'upload/company', 'image');
        if (! $saveLogo['status']) {
            $this->setError(['invoice_logo' => [$saveLogo['errors']]]);

            return false;
        }
        $saveDocs = null;
        if (! empty($request['image_stamp_docs']) && ! ($saveDocs = Helper::saveFiles($request['image_stamp_docs'], 'upload/company', 'image'))['status']) {
            $this->setError(['image_stamp_docs' => [$saveDocs['errors']]]);

            return false;
        }
        $request['logo'] = $saveLogo['path'];
        $request['docs'] = $saveDocs['path'] ?? null;

        $company = $this->companyRepository->create($request);
        $this->syncTranslations($company);

        return $company;
    }

    public function updateCompany($request)
    {
        if (! $this->validUpdated($request)) {
            return false;
        }
        $saveDocs = null;
        if (! empty($request['image_stamp_docs']) && ! ($saveDocs = Helper::saveFiles($request['image_stamp_docs'], 'upload/company', 'image'))['status']) {
            $this->setError(['image_stamp_docs' => [$saveDocs['errors']]]);

            return false;
        }
        $saveLogo = null;
        if (! empty($request['invoice_logo']) && ! $saveLogo = Helper::saveFiles($request['invoice_logo'], 'upload/company', 'image')) {
            $this->setError(['invoice_logo' => [$saveLogo['errors']]]);

            return false;
        }
        $request['logo'] = $saveLogo['path'] ?? null;
        $request['docs'] = $saveDocs['path'] ?? null;

        return $this->companyRepository->update($request);
    }

    private function validateCompany($request)
    {
        $rules = [
            'name' => ['required', 'string', 'max:250', 'unique:companies,name_company'],
            'companie_address' => ['required', 'string', 'max:200'],
            'companie_zip' => ['nullable', 'integer'],
            'company_vat' => ['nullable', 'string', 'max:255'],
            'companie_state' => ['required', 'integer', 'exists:area,id'],
            'companie_country' => ['required', 'integer', 'exists:countries,id'],
            'companie_city' => ['required', 'integer', 'exists:cities,id'],
            'companie_phone' => ['nullable', 'string', 'max:30'],
            'companie_fax' => ['nullable', 'string', 'max:30'],
            'companie_mobile' => ['nullable', 'string', 'max:30'],
            'companie_email' => ['nullable', 'string', 'max:100'],
            'invoice_logo' => ['required', 'regex:/^data:image\/(png|jpg|jpeg);base64,/'],
            'image_stamp_docs' => ['nullable', 'regex:/^data:image\/(png|jpg|jpeg);base64,/'],
            'image_stamp_docs_x' => ['nullable'],
            'image_stamp_docs_y' => ['nullable'],
            'image_stamp_docs_size' => ['nullable'],
            'apear_stamp' => ['nullable', 'integer'],
            'default_invoice_footer' => ['nullable', 'string'],
            'issuer_type' => ['nullable', 'in:B,P,F'],
            'issuer_id' => ['nullable', 'string', 'max:100'],
            'issuer_name' => ['nullable', 'string', 'max:100'],
            'issuer_branch_id' => ['nullable', 'integer'],
            'issuer_country' => ['nullable', 'string', 'max:10'],
            'issuer_governate' => ['nullable', 'string', 'max:255'],
            'issuer_region_city' => ['nullable', 'string', 'max:100'],
            'issuer_region_street' => ['nullable', 'string', 'max:255'],
            'issuer_building_number' => ['nullable', 'string', 'max:50'],
            'issuer_postal_code' => ['nullable', 'string', 'max:20'],
            'issuer_floor' => ['nullable', 'string', 'max:255'],
            'issuer_room' => ['nullable', 'string', 'max:255'],
            'issuer_landmark' => ['nullable', 'string', 'max:255'],
            'issuer_additional_information' => ['nullable', 'string', 'max:255'],
            'customers_payments' => ['nullable', 'integer'],
            'active' => ['required', 'in:0,1'],
            'accounting' => ['required', 'array'],
        ];
        $validator = Validator::make($request, $rules);
        $validator->after(function ($validator) use ($request) {
            $inputAccounts = $request['accounting'] ?? [];

            $activeCurrencies = Currency::where('status', 1)->pluck('name', 'id')->toArray();
            $cases = CompanyAccountMapping::cases();

            $expectedKeys = [];
            foreach ($cases as $case) {
                foreach ($activeCurrencies as $currencyId => $currencyName) {
                    $expectedKeys["{$case->value}_{$currencyId}"] = "{$case->value} + {$currencyName}";
                }
            }

            $sentKeys = [];
            $accountIds = [];
            $missingFieldsErrors = [];

            foreach ($inputAccounts as $index => $item) {
                $required = ['input_name', 'currency_id', 'tree_account_id'];
                $missing = array_diff($required, array_keys($item));

                if ($missing) {
                    $missingFieldsErrors[] = "العنصر رقم {$index} ناقص: ".implode(', ', $missing);

                    continue;
                }

                $sentKeys[] = "{$item['input_name']}_{$item['currency_id']}";
                $accountIds[] = $item['tree_account_id'];
            }

            if ($missingFieldsErrors) {
                foreach ($missingFieldsErrors as $err) {
                    $validator->errors()->add('accounting', $err);
                }

                return;
            }

            // تحقق من التوليفات الناقصة
            $missingKeys = array_diff(array_keys($expectedKeys), $sentKeys);

            if ($missingKeys) {
                $details = array_map(fn ($key) => 'الحساب المطلوب لـ '.$expectedKeys[$key], $missingKeys);
                $validator->errors()->add('accounting', 'لم يتم إرسال كل الحسابات المطلوبة: '.implode('، ', $details));

                return;
            }

            // تحقق من الحسابات في شجرة الحسابات
            $existing = TreeAccountingRepository::CheckTree($accountIds);
            $notFound = array_diff($accountIds, $existing);

            if ($notFound) {
                $validator->errors()->add(
                    'accounting',
                    'بعض الحسابات غير موجودة :'."\n".implode("\n", $notFound)
                );
            }
        });
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return true;
    }

    private function validUpdated($request)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:companies,id'],
            'name' => ['sometimes', 'string', 'max:250', 'unique:companies,name_company,'.$request['id']],
            'companie_address' => ['sometimes', 'string', 'max:200'],
            'companie_state' => ['sometimes', 'integer', 'exists:area,id'],
            'companie_country' => ['sometimes', 'integer', 'exists:countries,id'],
            'companie_city' => ['sometimes', 'integer', 'exists:cities,id'],
            'companie_zip' => ['sometimes', 'integer'],
            'company_vat' => ['sometimes', 'string', 'max:255'],
            'companie_phone' => ['sometimes', 'string', 'max:30'],
            'companie_fax' => ['sometimes', 'string', 'max:30'],
            'companie_mobile' => ['sometimes', 'string', 'max:30'],
            'companie_email' => ['sometimes', 'string', 'max:100'],
            'invoice_logo' => ['sometimes', 'regex:/^data:image\/(png|jpg|jpeg);base64,/'],
            'image_stamp_docs' => ['sometimes', 'regex:/^data:image\/(png|jpg|jpeg);base64,/'],
            'image_stamp_docs_x' => ['sometimes'],
            'image_stamp_docs_y' => ['sometimes'],
            'image_stamp_docs_size' => ['sometimes'],
            'apear_stamp' => ['sometimes', 'integer'],
            'default_invoice_footer' => ['sometimes', 'string'],
            'issuer_type' => ['sometimes', 'in:B,P,F'],
            'issuer_id' => ['sometimes', 'string', 'max:100'],
            'issuer_name' => ['sometimes', 'string', 'max:100'],
            'issuer_branch_id' => ['sometimes', 'integer'],
            'issuer_country' => ['sometimes', 'string', 'max:10'],
            'issuer_governate' => ['sometimes', 'string', 'max:255'],
            'issuer_region_city' => ['sometimes', 'string', 'max:100'],
            'issuer_region_street' => ['sometimes', 'string', 'max:255'],
            'issuer_building_number' => ['sometimes', 'string', 'max:50'],
            'issuer_postal_code' => ['sometimes', 'string', 'max:20'],
            'issuer_floor' => ['sometimes', 'string', 'max:255'],
            'issuer_room' => ['sometimes', 'string', 'max:255'],
            'issuer_landmark' => ['sometimes', 'string', 'max:255'],
            'issuer_additional_information' => ['sometimes', 'string', 'max:255'],
            'customers_payments' => ['sometimes', 'integer'],
            'active' => ['sometimes', 'in:0,1'],
            'accounting' => ['sometimes', 'array'],
        ];
        $validator = Validator::make($request, $rules);
        $validator->after(function ($validator) use ($request) {
            if (isset($request['accounting'])) {

                $inputAccounts = $request['accounting'] ?? [];

                $activeCurrencies = Currency::where('status', 1)->pluck('name', 'id')->toArray();
                $cases = CompanyAccountMapping::cases();

                $expectedKeys = [];
                foreach ($cases as $case) {
                    foreach ($activeCurrencies as $currencyId => $currencyName) {
                        $expectedKeys["{$case->value}_{$currencyId}"] = "{$case->value} + {$currencyName}";
                    }
                }

                $sentKeys = [];
                $accountIds = [];
                $missingFieldsErrors = [];

                foreach ($inputAccounts as $index => $item) {
                    $required = ['input_name', 'currency_id', 'tree_account_id'];
                    $missing = array_diff($required, array_keys($item));

                    if ($missing) {
                        $missingFieldsErrors[] = "العنصر رقم {$index} ناقص: ".implode(', ', $missing);

                        continue;
                    }

                    $sentKeys[] = "{$item['input_name']}_{$item['currency_id']}";
                    $accountIds[] = $item['tree_account_id'];
                }

                if ($missingFieldsErrors) {
                    foreach ($missingFieldsErrors as $err) {
                        $validator->errors()->add('accounting', $err);
                    }

                    return;
                }

                // تحقق من التوليفات الناقصة
                $missingKeys = array_diff(array_keys($expectedKeys), $sentKeys);

                if ($missingKeys) {
                    $details = array_map(fn ($key) => 'الحساب المطلوب لـ '.$expectedKeys[$key], $missingKeys);
                    $validator->errors()->add('accounting', 'لم يتم إرسال كل الحسابات المطلوبة: '.implode('، ', $details));

                    return;
                }

                // تحقق من الحسابات في شجرة الحسابات
                $existing = TreeAccountingRepository::CheckTree($accountIds);
                $notFound = array_diff($accountIds, $existing);

                if ($notFound) {
                    $validator->errors()->add(
                        'accounting',
                        'بعض الحسابات غير موجودة :'."\n".implode("\n", $notFound)
                    );
                }
            }
        });
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return true;
    }

    public function getCurrencyWithtree()
    {
        return Helper::getCurrencyWithtree(CompanyAccountMapping::class);
    }

    public function del(array $ids)
    {
        return $this->companyRepository->del($ids);
    }

    public function getByidWithTranslation($id)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:companies,id'],
        ];

        $validator = Validator::make(['id' => $id], $rules);

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }
        $company = $this->companyRepository->getByIdWithTranslation($id);

        return $this->syncTranslations($company, true);
    }

    public function getAllChanges($company)
    {
        return $this->companyRepository->getAllChanges($company);
    }
}
