<?php

namespace App\sys\Services\General;

use App\sys\Repository\General\CountryTranslationRepository;
use App\sys\Services;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;

class CountryTranslationService extends Services
{
    private $countryTranslation;

    public function __construct()
    {
        $this->countryTranslation = new CountryTranslationRepository;
    }

    public function save(array $data, $id)
    {
        $rules = ['country_id' => 'required|integer|exists:countries,id'];
        $combinations = [];
        $customErrors = [];

        foreach ($data as $index => $row) {
            $rowId = $row['id'] ?? null;
            $languageId = $row['language_id'] ?? null;
            // Validation rules
            $rules["$index.id"] = 'sometimes|nullable|exists:country_translation,id';
            $rules["$index.language_id"] = [
                Rule::requiredIf(empty($rowId)),
                'exists:languages,id',
                Rule::unique('country_translation', 'lang_id')
                    ->where(fn ($query) => $query->where('country_id', $id))
                    ->ignore($rowId),
            ];
            $rules["$index.name"] = 'nullable|string|max:255';

            // Check for duplicated combination
            if ($languageId) {
                $key = $id.'-'.$languageId;
                if (in_array($key, $combinations)) {
                    $customErrors["$index.language_id"] = 'Same language distorted with the same Country.';
                } else {
                    $combinations[] = $key;
                }
            }
        }

        $validator = Validator::make(array_merge($data, ['country_id' => $id]), $rules);

        // Inject custom duplicate error if any
        $validator->after(function ($validator) use ($customErrors) {
            foreach ($customErrors as $field => $message) {
                $validator->errors()->add($field, $message);
            }
        });

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->countryTranslation->save($data, $id);
    }

    public function get($id)
    {
        return $this->countryTranslation->get($id);
    }

    public function del($id)
    {
        $rules = [
            'id' => 'required|integer|exists:country_translation,id',
        ];
        $validator = \Validator::make(['id' => $id], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->countryTranslation->del($id);
    }
}
