<?php

namespace App\sys\Services\Invoice;

use App\Models\Airline;
use App\Models\Airports;
use App\Models\General\Service as ServiceModel;
use App\sys\Enums\ServicesType;
use App\sys\Services;
use Carbon\Carbon;
use Illuminate\Support\Facades\Log;

/**
 * Service class for parsing Amadeus Interface Record (A.I.R.) files
 * Extracts flight booking data and converts it to InvoiceServices format
 */
class AmadeusAirParserService extends Services
{
    /**
     * Parse A.I.R. file and return data array for InvoiceServices creation/update
     *
     * @param  string  $fileContent  The content of the A.I.R. file
     * @param  int  $profileId  The profile ID to associate with the invoice service
     * @return array|false Returns parsed data array ready for InvoiceServices or false on error
     */
    public function parseAirFile(string $fileContent, ?int $profileId = null): array|false
    {
        // التحقق من صحة تنسيق الملف - يجب أن يبدأ بـ AIR-BLK207
        if (! str_starts_with($fileContent, 'AIR-BLK207')) {
            $this->setError(['file' => ['Invalid A.I.R. file format. File must start with AIR-BLK207']]);

            return false;
        }

        // استخراج رقم PNR و office_id من الملف - PNR هو رقم الحجز الرئيسي
        $pnrAndOffice = $this->extractPnr($fileContent);
        $pnr = $pnrAndOffice['pnr'] ?? null;
        $officeId = $pnrAndOffice['office_id'] ?? null;
        if (! $pnr) {
            $this->setError(['file' => ['PNR number not found in file']]);

            return false;
        }

        // استخراج نوع السجل من السطر الأول - أنواع السجلات: 7A, MA, RF, NT, AMD
        $recordType = $this->extractRecordType($fileContent);
        if (! $recordType) {
            $this->setError(['file' => ['Record type not found in file']]);

            return false;
        }

        // البحث عن service_id بناءً على نوع السجل و unit_type
        // أولاً نبحث عن service بنوع flights و unit_type = recordType
        // إذا لم يوجد، نبحث عن أي service بنوع flights فقط
        $serviceId = $this->findServiceId($recordType);
        if (! $serviceId) {
            $this->setError(['service' => ['No service found for record type: '.$recordType.'. Please create a service with type=flights and unit_type='.$recordType]]);

            return false;
        }

        // استخراج معلومات الرحلات (Segments) من H- lines
        // Segments تحتوي على معلومات المطارات، شركات الطيران، التواريخ، أرقام الرحلات
        $segments = $this->extractSegments($fileContent);
        if (empty($segments)) {
            $this->setError(['segments' => ['No segments found in file']]);

            return false;
        }

        // تحديد نوع الرحلة - one_way (ذهاب فقط) أو round_trip (ذهاب وعودة)
        // يتم تحديده من B- line أو من عدد Segments
        $directionType = $this->determineDirectionType($fileContent, $segments);

        // استخراج عدد المسافرين الفعلي من الملف (قبل إنشاء المدخلات المتعددة)
        $passengerCounts = $this->extractPassengerCounts($fileContent);
        $adultCount = $passengerCounts['adult'] ?? 0;
        $childCount = $passengerCounts['child'] ?? 0;
        $infantCount = $passengerCounts['infant'] ?? 0;

        // استخراج معلومات المسافرين من I- lines و SSR DOCS
        // يحتوي على الأسماء، أرقام الجوازات، أرقام التذاكر
        $passengers = $this->extractPassengers($fileContent, $segments, $directionType);

        // استخراج معلومات الأسعار من K- lines
        // يحتوي على تكلفة الرحلة، الضرائب، الخصومات
        // نمرر عدد المسافرين الفعلي لتحديث pricing بشكل صحيح
        $pricing = $this->extractPricing($fileContent, $adultCount, $childCount, $infantCount);

        // استخراج تاريخ التذكرة: نحاول أولاً تاريخ التطبيق من D- أو O-، ثم أول Segment
        $ticketDate = $this->extractTicketDate($segments, $fileContent);

        // بناء مصفوفة البيانات الجاهزة لإنشاء/تحديث InvoiceService
        $requestData = [
            'travel_tourism_type' => ServicesType::FLIGHTS->value, // نوع الخدمة: flights
            'service_id' => $serviceId, // معرف الخدمة من جدول services
            'profile_id' => $profileId, // معرف البروفايل المرتبط (قد يكون null إذا لم يُرسل)
            'pnr_number' => $pnr, // رقم PNR
            'office_id' => $officeId, // رقم المكتب المستخرج من MUC1A (CAIEG24A0 مثلاً)
            'ticket_date' => $ticketDate, // تاريخ التذكرة
            'direction_type' => $directionType, // نوع الرحلة: one_way أو round_trip
            'segments' => $segments, // معلومات الرحلات
            'pax' => [
                'packs_price' => $pricing, // معلومات الأسعار
                'packs_details' => $passengers, // تفاصيل المسافرين
            ],
            // القيم المالية الأساسية: نمررها ليستخدمها FlightsHandler مباشرة
            'cost' => $pricing['adult']['cost'] ?? 0,
            'total_tax' => $pricing['adult']['tax'] ?? 0,
            'grand_total' => $pricing['adult']['total_cost'] ?? 0,
            // تخزين تجميعة/تفاصيل ضرائب الطيران في invoice_services.air_taxs (JSON)
            // (يتم cast إلى array في موديل InvoiceServices)
            'air_taxs' => $pricing['air_taxs'] ?? null,
        ];

        // استخراج فئة المقعد من الملف (F, P, A, R, J, C, D, Z, I, W, E, Y, B, H, K, M, L, V, S, N, Q, O, T, X, G)
        // القيمة الافتراضية: Y (Economy Class - Full Fare)
        $economyClass = $this->extractEconomyClass($fileContent, $segments);
        $requestData['economy_class'] = $economyClass ?: 'Y'; // استخدام Y كقيمة افتراضية إذا لم يتم العثور على قيمة

        // تسجيل معلومات عملية التحليل في Log للرجوع إليها لاحقاً
        Log::info('A.I.R. file parsed successfully', [
            'pnr' => $pnr,
            'record_type' => $recordType,
            'service_id' => $serviceId,
            'segments_count' => count($segments),
            'passengers_count' => count($passengers),
        ]);

        return $requestData;
    }

    /**
     * Extract PNR number from file content
     * PNR is found in MUC1A line (usually 4th line) - format: MUC1A 9URBJ3006;0101;CAIEG24A0;90256563;...
     *
     * @param  string  $content  File content
     * @return string|null PNR number or null if not found
     */
    private function extractPnr(string $content)
    {
        // تقسيم محتوى الملف إلى أسطر للبحث
        $lines = explode("\n", $content);

        // البحث عن سطر MUC1A الذي يحتوي على PNR
        // التنسيق: MUC1A 9URBJ3006;0101;CAIEG24A0;90256563;...
        // PNR موجود في الموقع الثالث بعد الفاصلة المنقوطة
        /**
         * foreach ($lines as $line) {
         * // البحث عن نمط: MUC1A متبوعاً بأي محتوى ثم فاصلة منقوطة ثم محتوى ثم فاصلة منقوطة ثم PNR (6-10 أحرف/أرقام)
         * if (preg_match('/MUC1A\s+[^;]*;[^;]*;([A-Z0-9]{6,10});/', $line, $matches)) {
         * return trim($matches[1]); // إرجاع PNR بعد تنظيف المسافات
         * }
         * }
         * */
        foreach ($lines as $line) {
            if (preg_match('/MUC1A\s+([A-Z0-9]+);[^;]*;([A-Z0-9]+)/', $line, $matches)) {
                return [
                    'pnr' => trim($matches[1]), // 88R4B3016
                    'office_id' => trim($matches[2]),  // CAIEG24A0
                ];
            }
        }
        // في حالة عدم العثور على PNR في MUC1A، البحث في B- line كبديل
        // التنسيق: B-TTP/T4/RT أو B-.../PNR
        foreach ($lines as $line) {
            if (preg_match('/B-[^\/]*\/([A-Z0-9]{6,10})/', $line, $matches)) {
                $pnr = trim($matches[1]); // رقم الـ PNR من سطر B-

                // نحاول أيضاً استخراج office_id بنفس طريقة سطر MUC1A (CAIEG24A0 مثلاً)
                $officeId = null;
                foreach ($lines as $innerLine) {
                    if (preg_match('/MUC1A\s+([A-Z0-9]+);[^;]*;([A-Z0-9]+)/', $innerLine, $mucMatches)) {
                        $officeId = trim($mucMatches[2]); // office_id من سطر MUC1A إن وجد
                        break;
                    }
                }

                return [
                    'pnr' => $pnr,
                    'office_id' => $officeId,
                ];
            }
        }

        return null; // لم يتم العثور على PNR
    }

    /**
     * Extract record type from file content
     * Record type is in the first line - format: AIR-BLK207;TYPE
     * Types: 7A (Booking), MA (Void), RF (Refund), NT (NDC Ticket), AMD (Amendment)
     *
     * @param  string  $content  File content
     * @return string|null Record type or null if not found
     */
    public function extractRecordType(string $content): ?string
    {
        // الحصول على السطر الأول من الملف
        $firstLine = explode("\n", $content)[0] ?? '';

        // البحث عن نمط: AIR-BLK207; متبوعاً بنوع السجل (أحرف وأرقام)
        if (preg_match('/AIR-BLK207;([A-Z0-9]+)/', $firstLine, $matches)) {
            return trim($matches[1]); // إرجاع نوع السجل بعد تنظيف المسافات
        }

        return null; // لم يتم العثور على نوع السجل
    }

    /**
     * Extract segments from H- lines
     * Format: H-001;002OCAI;CAIRO;BCN;BARCELONA;MS 0767 C C 21DEC1240 1600 21DEC;...
     * Each H- line represents one flight segment (go or return)
     *
     * @param  string  $content  File content
     * @return array Array of segment data with airport IDs, airline IDs, dates, times, flight numbers
     */
    private function extractSegments(string $content): array
    {
        $segments = []; // مصفوفة لحفظ معلومات Segments
        $lines = explode("\n", $content); // تقسيم الملف إلى أسطر

        $segmentIndex = 0; // عدّاد لتحديد go / return حسب الترتيب الفعلي في الملف
        // البحث عن جميع أسطر H- التي تمثل Segments
        foreach ($lines as $line) {
            // التحقق من أن السطر يبدأ بـ H- متبوعاً برقم Segment
            if (preg_match('/^H-(\d+);/', $line, $segmentMatch)) {
                $segmentNumber = (int) $segmentMatch[1]; // رقم Segment في الملف (قد لا يكون متسلسلاً)
                $segmentIndex++; // الترتيب الفعلي للـ segment في الملف

                // تحليل بيانات Segment من السطر
                // التنسيق: H-001;002OCAI;CAIRO;BCN;BARCELONA;MS    0767 C C 21DEC1240 1600 21DEC;...
                // 002O = رقم Segment، CAI = كود مطار المغادرة، BCN = كود مطار الوصول، MS = كود شركة الطيران، 0767 = رقم الرحلة
                // C C = economy_class (الحرف الأول هو فئة الحجز)
                // النمط: H-رقم;رقمOكود_مطار_مغادرة;اسم_مطار_مغادرة;كود_مطار_وصول;اسم_مطار_وصول;كود_شركة_طيران مسافات رقم_رحلة فئة_حجز فئة_حجز تاريخ...
                $economyClass = null; // تهيئة فئة الحجز
                $originCode = null;
                $destinationCode = null;
                $airlineCode = null;
                $flightNumber = null;

                if (preg_match('/H-\d+;(\d+)O([A-Z]{3});([^;]+);([A-Z]{3});([^;]+);([A-Z]{2})\s+(\d+)\s+([A-Z])\s+[A-Z]/', $line, $matches)) {
                    $originCode = trim($matches[2]); // كود مطار المغادرة (IATA code)
                    $destinationCode = trim($matches[4]); // كود مطار الوصول (IATA code)
                    $airlineCode = trim($matches[6]); // كود شركة الطيران (IATA code)
                    $flightNumber = trim($matches[7]); // رقم الرحلة
                    $economyClass = trim($matches[8]); // فئة الحجز (C, Y, F, etc.)
                } elseif (preg_match('/H-\d+;(\d+)O([A-Z]{3});([^;]+);([A-Z]{3});([^;]+);([A-Z]{2})\s+(\d+)/', $line, $matches)) {
                    // Fallback: استخراج البيانات الأساسية بدون economy_class
                    $originCode = trim($matches[2]);
                    $destinationCode = trim($matches[4]);
                    $airlineCode = trim($matches[6]);
                    $flightNumber = trim($matches[7]);
                    // محاولة استخراج economy_class من نمط مختلف
                    if (preg_match('/[A-Z]{2}\s+\d+\s+([A-Z])\s+[A-Z]/', $line, $classMatch)) {
                        $economyClass = trim($classMatch[1]);
                    }
                }

                // تخطي السطر إذا لم يتم استخراج البيانات الأساسية
                if (! $originCode || ! $destinationCode || ! $airlineCode || ! $flightNumber) {
                    continue;
                }

                // استخراج التاريخ والوقت من السطر
                // سنحاول العثور على أول نمط ddMMM+HHMM، وأيضاً نحتفظ بآخر تاريخ ddMMM كاحتياطي
                $dateTime = null;
                $time = null;

                $firstDateToken = null;
                $firstTimeToken = null;
                if (preg_match_all('/(\d{2}[A-Z]{3})(\d{4})/', $line, $dtMatches, PREG_SET_ORDER)) {
                    $firstDateToken = $dtMatches[0][1]; // مثل 21DEC
                    $firstTimeToken = $dtMatches[0][2]; // مثل 1240
                }

                $lastDateOnly = null;
                if (preg_match_all('/(\d{2}[A-Z]{3})(?!\d)/', $line, $dOnlyMatches, PREG_SET_ORDER)) {
                    $last = end($dOnlyMatches);
                    $lastDateOnly = $last[1]; // آخر تاريخ موجود في السطر (بدون وقت)
                }

                if ($firstDateToken) {
                    $dateTime = $this->parseAmadeusDate($firstDateToken);
                    $time = $firstTimeToken;
                } elseif ($lastDateOnly) {
                    $dateTime = $this->parseAmadeusDate($lastDateOnly);
                }

                // إذا فشل استخراج التاريخ، استخدام تاريخ اليوم كقيمة افتراضية
                // هذا يضمن أن date لن يكون null وبالتالي لن يفشل التحقق
                if (! $dateTime) {
                    $dateTime = Carbon::now();
                    Log::warning('Failed to extract date from segment line, using current date', [
                        'line' => $line,
                        'segment_number' => $segmentNumber,
                    ]);
                }

                // تحديد نوع Segment: go (ذهاب) أو return (عودة)
                // نعتمد على ترتيب الظهور في الملف وليس رقم السطر داخل H-
                $type = $segmentIndex === 1 ? 'go' : 'return';

                // البحث عن معرف مطار المغادرة من قاعدة البيانات باستخدام كود IATA
                $originAirportId = $this->findAirportId($originCode);

                // البحث عن معرف مطار الوصول من قاعدة البيانات باستخدام كود IATA
                $destinationAirportId = $this->findAirportId($destinationCode);

                // البحث عن معرف شركة الطيران من قاعدة البيانات باستخدام كود IATA
                $airlineId = $this->findAirlineId($airlineCode);

                // إضافة بيانات Segment إلى المصفوفة
                $segments[] = [
                    'origin_airport' => $originAirportId, // معرف مطار المغادرة
                    'destination_airport' => $destinationAirportId, // معرف مطار الوصول
                    'date' => $dateTime->format('Y-m-d'), // تاريخ الرحلة بصيغة Y-m-d (مضمون أن يكون موجوداً)
                    'time' => $time, // وقت الرحلة (HHMM)
                    'type' => $type, // نوع Segment: go أو return
                    'flight_number' => $flightNumber, // رقم الرحلة
                    'flight_company' => $airlineId, // معرف شركة الطيران
                    'economy_class' => $economyClass ?: null, // فئة الحجز (C, Y, F, etc.) أو null إذا لم يتم العثور عليها
                ];
            }
        }

        return $segments; // إرجاع جميع Segments المستخرجة
    }

    /**
     * Extract passenger counts from I- lines
     * Counts unique passengers by type (Adult, Child, Infant) from the file
     *
     * @param  string  $content  File content
     * @return array Array with 'adult', 'child', 'infant' counts
     */
    private function extractPassengerCounts(string $content): array
    {
        $lines = explode("\n", $content);
        $passengerNames = [];

        // استخراج أسماء المسافرين من أسطر I-
        foreach ($lines as $line) {
            if (preg_match('/^I-(\d+);([^;]+);/', $line, $matches)) {
                $passengerNumber = (int) $matches[1];
                $nameBlock = trim($matches[2]); // مثال: 01TADROS/MINA MR

                // إزالة أي أرقام بادئة
                $nameBlock = ltrim($nameBlock, '0123456789');

                // تقسيم إلى أجزاء وفصل اللقب الشرفي (MR/MRS/...)
                $parts = preg_split('/\s+/', $nameBlock);
                $titleRaw = count($parts) > 1 ? array_pop($parts) : null;
                $title = $titleRaw ? $this->normalizeTitle($titleRaw) : null;

                // الباقي يفترض أنه LAST/FIRST
                $nameCore = implode(' ', $parts);
                $nameSegments = explode('/', $nameCore);
                $lastName = trim($nameSegments[0] ?? '');
                $firstName = trim($nameSegments[1] ?? '');

                // حفظ معلومات المسافر (افتراضي: Adult)
                $key = $firstName.'|'.$lastName;
                if (! isset($passengerNames[$key])) {
                    $passengerNames[$key] = 'Adult'; // افتراضي: Adult
                }
            }
        }

        // حساب عدد المسافرين حسب النوع
        $adultCount = count($passengerNames); // افتراضي: جميع المسافرين بالغين
        $childCount = 0;
        $infantCount = 0;

        // يمكن تحسين هذا لاستخراج نوع المسافر من الملف إذا كان متوفراً
        // حالياً نعتبر جميع المسافرين بالغين

        return [
            'adult' => $adultCount,
            'child' => $childCount,
            'infant' => $infantCount,
        ];
    }

    /**
     * Extract passengers from I- lines and SSR DOCS
     * Format: I-001;01MOHAREB/SONIA MRS;;APAMAL//CAI +201224989679...
     * SSR DOCS MS  HK1/P/BEL/GA1762772/BEL/24AUG66/F/29MAY29/MOHAREB/SONIA;P1
     *
     * @param  string  $content  File content
     * @param  array  $segments  Array of segments to determine passenger direction
     * @param  string  $directionType  Direction type: one_way or round_trip
     * @return array Array of passenger data formatted for packs_details
     */
    private function extractPassengers(string $content, array $segments, string $directionType): array
    {
        $passengers = []; // مصفوفة لحفظ معلومات المسافرين
        $lines = explode("\n", $content); // تقسيم الملف إلى أسطر

        // استخراج أسماء المسافرين من أسطر I-
        // التنسيق الشائع: I-001;01LAST/FIRST TITLE;;
        // مثال: I-001;01TADROS/MINA MR;;
        $passengerNames = [];
        foreach ($lines as $line) {
            if (preg_match('/^I-(\d+);([^;]+);/', $line, $matches)) {
                $passengerNumber = (int) $matches[1]; // رقم المسافر
                $nameBlock = trim($matches[2]); // مثل: 01TADROS/MINA MR

                // إزالة أي أرقام بادئة (مثل 01)
                $nameBlock = ltrim($nameBlock, '0123456789');

                // فصل اللقب والاسم الأول عن اللقب الشرفي
                $parts = preg_split('/\s+/', $nameBlock);
                $titleRaw = count($parts) > 1 ? array_pop($parts) : null; // آخر جزء هو اللقب الشرفي (MR/MRS/...)
                $title = $titleRaw ? $this->normalizeTitle($titleRaw) : null;

                // ما تبقى يفترض أنه LAST/FIRST
                $nameCore = implode(' ', $parts);
                $nameSegments = explode('/', $nameCore);
                $lastName = trim($nameSegments[0] ?? '');
                $firstName = trim($nameSegments[1] ?? '');

                $passengerNames[$passengerNumber] = [
                    'title' => $title,
                    'first_name' => $firstName,
                    'last_name' => $lastName,
                    'type' => 'Adult', // افتراضي، يمكن تحديثه لاحقاً
                ];
            }
        }

        // استخراج أرقام الجوازات من أسطر SSR DOCS
        // التنسيق: SSR DOCS MS  HK1/P/BEL/GA1762772/BEL/24AUG66/F/29MAY29/MOHAREB/SONIA;P1
        // GA1762772 = رقم الجواز
        $passportNumbers = [];
        foreach ($lines as $line) {
            // البحث عن نمط: SSR DOCS متبوعاً بمعلومات الجواز
            if (preg_match('/SSR DOCS\s+[^\/]+\/([A-Z])\/([A-Z0-9]+)\//', $line, $matches)) {
                $passportNumbers[] = trim($matches[2]); // إضافة رقم الجواز إلى المصفوفة
            }
        }

        // استخراج أرقام التذاكر من أسطر T-
        // التنسيق: T-K077-6504647865
        // K077-6504647865 = رقم التذكرة
        $ticketNumbers = [];
        foreach ($lines as $line) {
            // البحث عن نمط: T- متبوعاً برقم التذكرة
            if (preg_match('/^T-([A-Z]\d+-\d+)/', $line, $matches)) {
                $ticketNumbers[] = trim($matches[1]); // إضافة رقم التذكرة إلى المصفوفة
            }
        }

        // بناء مصفوفة المسافرين النهائية
        $passengerIndex = 0; // فهرس المسافر الحالي
        foreach ($passengerNames as $passenger) {
            // الحصول على رقم الجواز والتذكرة للمسافر الحالي (إن وجدا)
            $passportNumber = $passportNumbers[$passengerIndex] ?? null;
            $ticketNumber = $ticketNumbers[$passengerIndex] ?? null;

            // في حالة الرحلة ذهاب وعودة، ننشئ مدخلين لكل مسافر (ذهاب وعودة)
            if ($directionType === 'round_trip') {
                // مدخل للذهاب
                $passengers[] = [
                    'title' => $passenger['title'], // اللقب
                    'first_name' => $passenger['first_name'], // الاسم الأول
                    'last_name' => $passenger['last_name'], // اللقب
                    'passport_number' => $passportNumber, // رقم الجواز
                    'ticket_number' => $ticketNumber, // رقم التذكرة
                    'type' => $passenger['type'], // نوع المسافر
                    'direction_type' => 'go', // نوع الاتجاه: go (ذهاب)
                    'emd' => 'NONE', // EMD (افتراضي: NONE)
                ];

                // مدخل للعودة
                $passengers[] = [
                    'title' => $passenger['title'],
                    'first_name' => $passenger['first_name'],
                    'last_name' => $passenger['last_name'],
                    'passport_number' => $passportNumber,
                    'ticket_number' => $ticketNumber,
                    'type' => $passenger['type'],
                    'direction_type' => 'back', // نوع الاتجاه: back (عودة)
                    'emd' => 'NONE',
                ];
            } else {
                // في حالة الرحلة ذهاب فقط، ننشئ مدخل واحد فقط
                $passengers[] = [
                    'title' => $passenger['title'],
                    'first_name' => $passenger['first_name'],
                    'last_name' => $passenger['last_name'],
                    'passport_number' => $passportNumber,
                    'ticket_number' => $ticketNumber,
                    'type' => $passenger['type'],
                    'direction_type' => 'go', // نوع الاتجاه: go (ذهاب فقط)
                    'emd' => 'NONE',
                ];
            }

            $passengerIndex++; // الانتقال للمسافر التالي
        }

        return $passengers; // إرجاع جميع المسافرين
    }

    /**
     * Extract pricing from K- lines
     * Format: KN-IEGP49999.00   ;;;;;;;;;;;;EGP66195.00   ;;;
     * KNTI; EGP1083.00  YQ AC; EGP1288.00  YQ AD; ...
     * KN- = Net fare, KS- = Selling fare, KNTI/KSTI = Tax breakdown
     *
     * @param  string  $content  File content
     * @param  int  $adultCount  Number of adult passengers
     * @param  int  $childCount  Number of child passengers
     * @param  int  $infantCount  Number of infant passengers
     * @return array Pricing data formatted for packs_price (adult, child, infant)
     */
    private function extractPricing(string $content, int $adultCount = 0, int $childCount = 0, int $infantCount = 0): array
    {
        $lines = explode("\n", $content); // تقسيم الملف إلى أسطر

        // تهيئة مصفوفة الأسعار بقيم افتراضية
        // تحتوي على معلومات للكبار (adult)، الأطفال (child)، الرضع (infant)
        $pricing = [
            'adult' => [
                'count' => 0, // عدد الكبار
                'cost' => 0, // تكلفة الرحلة للكبار
                'tax' => 0, // الضرائب للكبار
                'earned_discount' => 0, // الخصم المكتسب
                'granted_discount' => 0, // الخصم الممنوح
                'total_emd' => 0, // إجمالي EMD
                'total_cost' => 0, // إجمالي التكلفة
            ],
            'child' => [
                'count' => 0,
                'cost' => 0,
                'tax' => 0,
                'earned_discount' => 0,
                'granted_discount' => 0,
                'total_emd' => 0,
                'total_cost' => 0,
            ],
            'infant' => [
                'count' => 0,
                'cost' => 0,
                'tax' => 0,
                'earned_discount' => 0,
                'granted_discount' => 0,
                'total_emd' => 0,
                'total_cost' => 0,
            ],
        ];

        $baseFare = 0; // السعر الأساسي للرحلة
        $totalTax = 0; // إجمالي الضرائب
        $fareTotal = 0; // الإجمالي (fare + tax) من سطر KN/KS أو K-

        // تفاصيل الضرائب (tax breakdown) لاستخدامها في air_taxs داخل invoice_services
        $taxItems = [];

        // البحث عن أسطر الأسعار:
        // - KN- / KS- (التنسيق السابق)
        // - K- (كما في ملفات أماديوس التي لا تحتوي KN/KS)
        // - RFDC (سطر تفاصيل الاسترداد)
        // - KRF (ضرائب الاسترداد)
        // - KFTF / KFTR (ضرائب أماديوس بتنسيقات مختلفة)
        // نلتقط أيضاً آخر قيمة عملة/مبلغ في السطر لاستخدامها كمجموع محلي إن وُجد
        foreach ($lines as $line) {
            $lastCurrencyAmount = null;

            // استخراج السعر الأساسي من أسطر KN-I أو KS-I
            if (preg_match('/^(KN|KS)-I([A-Z]{3})([\d.]+)/', $line, $matches)) {
                $baseFare = (float) $matches[3];
                $pricing['adult']['cost'] = $baseFare;
                // آخر مبلغ في نفس السطر غالباً هو الإجمالي المحلي
                if (preg_match_all('/([A-Z]{3})([\d.]+)/', $line, $allAmounts, PREG_SET_ORDER)) {
                    $last = end($allAmounts);
                    $fareTotal = (float) $last[2];
                }
            }

            // استخراج السعر الأساسي من سطر K- (مثال: K-RUSD1114.00 ;EGP ... EGP14249.80 ...)
            if (preg_match('/^K-[A-Z]?([A-Z]{3})([\d.]+)/', $line, $matches)) {
                $baseFare = (float) $matches[2];
                $pricing['adult']['cost'] = $baseFare;
                if (preg_match_all('/([A-Z]{3})([\d.]+)/', $line, $allAmounts, PREG_SET_ORDER)) {
                    $last = end($allAmounts);
                    $fareTotal = (float) $last[2];
                }
            }

            // تحليل أسطر KNTI أو KSTI للحصول على تفاصيل الضرائب (التنسيق القديم)
            if (preg_match('/^(KN|KS)TI;/', $line)) {
                if (preg_match_all('/\s+([A-Z]{3})([\d.]+)\s+([A-Z]{2})\s+([A-Z]{2})/', $line, $taxMatches, PREG_SET_ORDER)) {
                    foreach ($taxMatches as $taxMatch) {
                        $taxAmount = (float) $taxMatch[2];
                        $totalTax += $taxAmount;
                        $taxItems[] = [
                            'source' => 'KNTI/KSTI',
                            'currency' => $taxMatch[1] ?? null,
                            'amount' => $taxAmount,
                            'tax_code' => $taxMatch[3] ?? null,
                            'tax_type' => $taxMatch[4] ?? null,
                        ];
                    }
                }
            }

            // تحليل سطر KFTR (ضرائب أماديوس البديلة)
            if (preg_match('/^KFTR;/', $line)) {
                // البحث عن أنماط: عملة + مبلغ + كودين (مثال: OEGP585.00   F6 TO)
                if (preg_match_all('/[A-Z]{0,1}([A-Z]{3})([\d.]+)\s+([A-Z0-9]{2})\s+([A-Z0-9]{2})/', $line, $taxMatches, PREG_SET_ORDER)) {
                    foreach ($taxMatches as $taxMatch) {
                        $taxAmount = (float) $taxMatch[2];
                        $totalTax += $taxAmount;
                        $taxItems[] = [
                            'source' => 'KFTR',
                            'currency' => $taxMatch[1] ?? null,
                            'amount' => $taxAmount,
                            'tax_code' => $taxMatch[3] ?? null,
                            'tax_type' => $taxMatch[4] ?? null,
                        ];
                    }
                }
            }

            // تحليل سطر KFTF (تنسيق بديل للضرائب، كما في CAIEG24A0_AIR_0037)
            if (preg_match('/^KFTF;/', $line)) {
                if (preg_match_all('/\s([A-Z]{3})([\d.]+)\s+([A-Z0-9]{2})\s+([A-Z0-9]{2})/', $line, $taxMatches, PREG_SET_ORDER)) {
                    foreach ($taxMatches as $taxMatch) {
                        $taxAmount = (float) $taxMatch[2];
                        $totalTax += $taxAmount;
                        $taxItems[] = [
                            'source' => 'KFTF',
                            'currency' => $taxMatch[1] ?? null,
                            'amount' => $taxAmount,
                            'tax_code' => $taxMatch[3] ?? null,
                            'tax_type' => $taxMatch[4] ?? null,
                        ];
                    }
                }
            }

            // تحليل سطر KRF (ضرائب الاسترداد)
            if (preg_match('/^KRF\s*;/', $line)) {
                if (preg_match_all('/\s([A-Z]{3})([\d.]+)/', $line, $taxMatches, PREG_SET_ORDER)) {
                    foreach ($taxMatches as $taxMatch) {
                        $taxAmount = (float) $taxMatch[2];
                        $totalTax += $taxAmount;
                        $taxItems[] = [
                            'source' => 'KRF',
                            'currency' => $taxMatch[1] ?? null,
                            'amount' => $taxAmount,
                            'tax_code' => null,
                            'tax_type' => null,
                        ];
                    }
                }
            }

            // تحليل سطر RFDC (تفاصيل استرداد تتضمن المبالغ الإجمالية)
            // مثال: RFDC;02DEC25;I;EGP93442.00;0.00;93442.00;;;;;;XT22965.00;116407.00;21DEC25
            if (preg_match('/^RFDC;/', $line)) {
                // استخدم أول مبلغ كـ base fare، وآخر مبلغ كإجمالي، والفرق كضريبة تقريبية إن وجد
                if (preg_match_all('/([A-Z]{3})([\d.]+)/', $line, $allAmounts, PREG_SET_ORDER)) {
                    $first = $allAmounts[0];
                    $last = end($allAmounts);
                    $baseFare = (float) $first[2];
                    $fareTotal = (float) $last[2];
                    $pricing['adult']['cost'] = $baseFare;
                }
            }
        }

        // حفظ معلومات الضرائب والسعر الإجمالي
        // تحديث عدد المسافرين دائماً (حتى لو كان 0)
        $pricing['adult']['count'] = $adultCount;
        $pricing['child']['count'] = $childCount;
        $pricing['infant']['count'] = $infantCount;

        // حفظ الأسعار الإجمالية (بدون تقسيم على عدد المسافرين)
        // سيتم تقسيمها لاحقاً في calculatePerPersonCost عند الحاجة
        if ($baseFare > 0) {
            $pricing['adult']['cost'] = $baseFare; // السعر الإجمالي للكبار
            // إذا لم تُحتسب الضريبة لكنها ضمن إجمالي محلي، نحاول استنتاجها
            if ($totalTax === 0 && $fareTotal > $baseFare) {
                $totalTax = $fareTotal - $baseFare;
            }
            $pricing['adult']['tax'] = $totalTax; // الضرائب الإجمالية للكبار
            // إذا كان لدينا fareTotal نستخدمه، وإلا baseFare + totalTax
            $pricing['adult']['total_cost'] = $fareTotal > 0
                ? $fareTotal
                : $baseFare + $totalTax;
        } else {
            // إذا لم يتم العثور على سعر أساسي
            $pricing['adult']['tax'] = $totalTax;
            $pricing['adult']['total_cost'] = $pricing['adult']['cost'] + $pricing['adult']['tax'];
        }

        // تجهيز payload للـ air_taxs (تجميع الضرائب)
        // نخزن items + aggregated + total (كما هو محسوب)
        $aggregated = [];
        foreach ($taxItems as $item) {
            $code = $item['tax_code'] ?? 'NA';
            $type = $item['tax_type'] ?? 'NA';
            $key = $code.'-'.$type;
            $aggregated[$key] = ($aggregated[$key] ?? 0) + (float) ($item['amount'] ?? 0);
        }
        ksort($aggregated);

        $shouldStoreAirTaxes = ((float) $totalTax) > 0 || ! empty($taxItems);
        $pricing['air_taxs'] = $shouldStoreAirTaxes ? [
            'total_tax' => (float) $totalTax,
            // نضمن نفس الشكل دائماً: keys ثابتة، والقيم ممكن تكون null
            'items' => ! empty($taxItems) ? $taxItems : null,
            'aggregated' => ! empty($aggregated) ? $aggregated : null,
        ] : null;

        // تحديث عدد الأطفال والرضع
        $pricing['child']['count'] = $childCount;
        $pricing['infant']['count'] = $infantCount;

        return $pricing; // إرجاع معلومات الأسعار
    }

    /**
     * Find service ID based on record type
     * First searches for service with type='flights' and unit_type=$recordType
     * If not found, searches for any service with type='flights'
     *
     * @param  string  $recordType  Record type (7A, MA, RF, NT, AMD)
     * @return int|null Service ID or null if not found
     */
    private function findServiceId(string $recordType): ?int
    {
        // المحاولة الأولى: البحث عن service بنوع flights و unit_type = recordType
        // مثال: type='flights' و unit_type='7A'
        // ملاحظة: العمود في قاعدة البيانات اسمه 'type'
        $service = ServiceModel::where('type', ServicesType::FLIGHTS->value)
            ->where('unit_type', $recordType) // البحث بـ unit_type المحدد
            ->where('is_active', 1) // فقط الخدمات النشطة
            ->first();

        // إذا تم العثور على service، إرجاع معرفه
        if ($service) {
            return $service->id;
        }

        // المحاولة الثانية (Fallback): البحث عن أي service بنوع flights فقط (بدون unit_type)
        // هذا في حالة عدم وجود service محدد بـ unit_type
        $service = ServiceModel::where('type', ServicesType::FLIGHTS->value)
            ->where('is_active', 1) // فقط الخدمات النشطة
            ->first();

        return $service?->id; // إرجاع معرف Service أو null إذا لم يوجد
    }

    /**
     * Find airport ID by IATA code
     * Searches in airports table for matching code
     *
     * @param  string  $iataCode  IATA airport code (e.g., CAI, BCN, JFK)
     * @return int|null Airport ID or null if not found
     */
    private function findAirportId(string $iataCode): ?int
    {
        // البحث عن معرف المطار من جدول airports باستخدام كود IATA
        // مثال: البحث عن CAI لإيجاد معرف مطار القاهرة
        return Airports::where('code', $iataCode)->value('id');
    }

    /**
     * Find airline ID by code
     * Searches in airlines table for matching code
     *
     * @param  string  $airlineCode  IATA airline code (e.g., MS, LH, BA)
     * @return int|null Airline ID or null if not found
     */
    private function findAirlineId(string $airlineCode): ?int
    {
        // البحث عن معرف شركة الطيران من جدول airlines باستخدام كود IATA
        // مثال: البحث عن MS لإيجاد معرف مصر للطيران
        return Airline::where('code', $airlineCode)->value('id');
    }

    /**
     * Determine direction type (one_way or round_trip) from B- line or segments count
     * Checks B- line for RT indicator, or counts segments
     *
     * @param  string  $content  File content
     * @param  array  $segments  Array of segments
     * @return string Direction type: 'one_way' or 'round_trip'
     */
    private function determineDirectionType(string $content, array $segments): string
    {
        $lines = explode("\n", $content); // تقسيم الملف إلى أسطر

        // التحقق من سطر B- لوجود مؤشر RT (Round Trip)
        // التنسيق: B-TTP/T4/RT - RT يشير إلى رحلة ذهاب وعودة
        foreach ($lines as $line) {
            if (preg_match('/B-[^\/]*\/RT/', $line)) {
                return 'round_trip'; // رحلة ذهاب وعودة
            }
        }

        // إذا لم يتم العثور على RT في B- line، نتحقق من عدد Segments
        // إذا كان هناك أكثر من Segment واحد، فهي رحلة ذهاب وعودة
        if (count($segments) > 1) {
            return 'round_trip'; // رحلة ذهاب وعودة
        }

        return 'one_way'; // رحلة ذهاب فقط
    }

    /**
     * Extract ticket date from segments
     * Uses the date from the first segment, or current date if not available
     *
     * @param  array  $segments  Array of segments
     * @return string Ticket date in Y-m-d format
     */
    private function extractTicketDate(array $segments, string $content): string
    {
        // 1) محاولة قراءة تاريخ التطبيق من سطر D- (تنسيق ddmmyy مثل 251208)
        $lines = explode("\n", $content);
        foreach ($lines as $line) {
            if (str_starts_with($line, 'D-')) {
                // مثال: D-251208;251208;251208
                $parts = explode(';', substr($line, 2));
                $raw = trim($parts[0] ?? '');
                if (preg_match('/^(\\d{2})(\\d{2})(\\d{2})$/', $raw, $m)) {
                    $day = (int) $m[1];
                    $month = (int) $m[2];
                    $year = (int) $m[3];
                    // نفترض سنة 2000+ للسنوات المكونة من رقمين
                    $year += $year < 70 ? 2000 : 1900;

                    return Carbon::create($year, $month, $day)->format('Y-m-d');
                }
            }
        }

        // 2) محاولة من سطر O- (مثل 12DEC12DEC;16DEC16DEC)
        foreach ($lines as $line) {
            if (str_starts_with($line, 'O-')) {
                if (preg_match('/O-([0-9]{2}[A-Z]{3})/', $line, $m)) {
                    $date = $this->parseAmadeusDate($m[1]);
                    if ($date) {
                        return $date->format('Y-m-d');
                    }
                }
            }
        }

        // 3) استخدام تاريخ أول Segment
        if (! empty($segments) && isset($segments[0]['date'])) {
            return $segments[0]['date'];
        }

        // 4) الافتراضي: اليوم
        return now()->format('Y-m-d');
    }

    /**
     * Extract economy class from segments or file content
     * Looks for class indicators in segments: F, P, A, R, J, C, D, Z, I, W, E, Y, B, H, K, M, L, V, S, N, Q, O, T, X, G
     *
     * @param  string  $content  File content
     * @param  array  $segments  Array of segments (each segment may contain 'economy_class')
     * @return string|null Economy class code (F, P, A, R, J, C, D, Z, I, W, E, Y, B, H, K, M, L, V, S, N, Q, O, T, X, G) or null
     */
    private function extractEconomyClass(string $content, array $segments): ?string
    {
        // القيم المسموحة لـ economy_class
        $allowedClasses = [
            'F', 'P', 'A', 'R', // First Class
            'J', 'C', 'D', 'Z', 'I', // Business Class
            'W', 'E', // Premium Economy
            'Y', 'B', 'H', 'K', 'M', 'L', 'V', 'S', 'N', 'Q', 'O', 'T', 'X', 'G', // Economy Class
        ];

        // البحث عن فئة الحجز في Segments
        // إذا كان Segment يحتوي على economy_class، نستخدمه
        foreach ($segments as $segment) {
            if (isset($segment['economy_class']) && in_array($segment['economy_class'], $allowedClasses, true)) {
                return $segment['economy_class']; // إرجاع أول فئة حجز صالحة نجدها
            }
        }

        // محاولة استخراج فئة الحجز مباشرة من محتوى الملف (من أسطر H-)
        // التنسيق: MS 0767 C C 21DEC1240 1600 21DEC
        // حيث C C هو فئة الحجز (الحرف الأول)
        $lines = explode("\n", $content);
        foreach ($lines as $line) {
            // البحث عن نمط: شركة طيران + رقم رحلة + فئة حجز (حرف واحد) + حرف آخر + تاريخ
            // مثال: MS 0767 C C 21DEC1240
            if (preg_match('/[A-Z]{2}\s+\d+\s+([A-Z])\s+[A-Z]\s+\d{2}[A-Z]{3}/', $line, $classMatches)) {
                $classCode = trim($classMatches[1]);
                if (in_array($classCode, $allowedClasses, true)) {
                    return $classCode; // إرجاع فئة الحجز المستخرجة
                }
            }
        }

        // القيمة الافتراضية: Y (Economy Class - Full Fare)
        return 'Y';
    }

    /**
     * Parse Amadeus date format (e.g., 21DEC2024 or 21DEC24 or 21DEC)
     * Converts to Carbon date object
     *
     * @param  string  $dateString  Date string in Amadeus format
     * @return Carbon|null Carbon date object or null if parsing fails
     */
    private function parseAmadeusDate(string $dateString): ?Carbon
    {
        // التنسيق: DDMMMYYYY أو DDMMMYY أو DDMMM
        // مثال: 21DEC2024 أو 21DEC24 أو 21DEC
        // البحث عن نمط: رقمين (اليوم) + 3 أحرف (الشهر) + 0-4 أرقام (السنة - اختياري)
        if (preg_match('/(\d{2})([A-Z]{3})(\d{2,4})?/', $dateString, $matches)) {
            $day = $matches[1]; // اليوم (21)
            $month = $matches[2]; // الشهر (DEC)
            $year = $matches[3] ?? ''; // السنة (2024 أو 24 أو فارغة)

            // إذا لم تكن هناك سنة، استخدام السنة الحالية
            if (empty($year)) {
                $year = now()->format('Y'); // السنة الحالية
            } elseif (strlen($year) === 2) {
                // إذا كانت السنة مكونة من رقمين، إضافة 20 في البداية
                // مثال: 24 تصبح 2024
                $year = '20'.$year;
            }

            // بناء نص التاريخ بصيغة يمكن لـ Carbon فهمها
            // التنسيق: "21 DEC 2024"
            $dateStr = $day.' '.$month.' '.$year;

            // محاولة تحويل التاريخ إلى Carbon object
            // إذا فشل التحويل، سيتم إرجاع null
            $parsedDate = Carbon::createFromFormat('d M Y', $dateStr, false);

            // التحقق من صحة التاريخ المحلل
            if ($parsedDate && $parsedDate->format('d M Y') === $dateStr) {
                return $parsedDate;
            }

            // في حالة فشل التحويل، تسجيل تحذير في Log
            Log::warning('Failed to parse Amadeus date', ['date' => $dateString, 'parsed' => $dateStr]);
        }

        return null; // إرجاع null في حالة فشل التحليل
    }

    /**
     * Normalize title (MR, MRS, MISS, etc.)
     * Converts various title formats to lowercase standard format
     *
     * @param  string  $title  Title string (e.g., MR, MRS, MISS, MS)
     * @return string Normalized title in lowercase (e.g., mr, mrs, miss, ms)
     */
    private function normalizeTitle(string $title): string
    {
        // تحويل العنوان إلى أحرف كبيرة وإزالة المسافات
        $title = strtoupper(trim($title));

        // خريطة للعناوين الشائعة وتحويلها إلى صيغة موحدة
        $titleMap = [
            'MR' => 'mr', // السيد
            'MRS' => 'mrs', // السيدة
            'MISS' => 'miss', // الآنسة
            'MS' => 'ms', // السيدة/الآنسة
        ];

        // إذا كان العنوان موجوداً في الخريطة، إرجاع القيمة الموحدة
        // إذا لم يكن موجوداً، إرجاع العنوان بحروف صغيرة
        return $titleMap[$title] ?? strtolower($title);
    }

    /**
     * Save A.I.R. file to public directory
     * Stores file in public/air/ directory with PNR and timestamp
     *
     * @param  string  $fileContent  File content to save
     * @param  string  $pnr  PNR number to include in filename
     * @return string Saved file path relative to public directory (air/PNR_TIMESTAMP.txt)
     */
    public function saveAirFile(string $fileContent, string $pnr): string
    {
        // بناء اسم الملف: air/PNR_TIMESTAMP.txt
        // مثال: air/CAIEG24A0_20241221143025.txt
        $filename = $pnr.'_'.now()->format('YmdHis').'.txt';
        $directory = public_path('air');
        $filePath = $directory.'/'.$filename;

        // التأكد من وجود المجلد، وإنشاؤه إذا لم يكن موجوداً
        if (! is_dir($directory)) {
            mkdir($directory, 0755, true);
        }

        // حفظ الملف في public/air/
        file_put_contents($filePath, $fileContent);

        // إرجاع المسار النسبي من public (air/PNR_TIMESTAMP.txt)
        return 'air/'.$filename;
    }
}
