<?php

namespace App\sys\Services\Profile;

use App\sys\Repository\Accounting\UnpostedCollectionsRepository;
use App\sys\Repository\Invoice\InvoiceServicesRepository;
use App\sys\Repository\Profile\AccommodationReservationRepository;
use App\sys\Repository\Profile\ProfileRepository;
use App\sys\Services;
use Carbon\Carbon;
use Illuminate\Support\Facades\Validator;

class ProfileService extends Services
{
    protected ProfileRepository $profileRepository;

    private $accmodationRepository;

    private $invoiceServiceRepository;

    public function __construct(ProfileRepository $profileRepository)
    {
        $this->profileRepository = $profileRepository;
        $this->invoiceServiceRepository = new InvoiceServicesRepository;
        $this->accmodationRepository = new AccommodationReservationRepository;
    }

    public function getPaginated()
    {
        return $this->profileRepository->getPaginated();
    }

    public function getById($id)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:pr_profile,id'],
        ];
        $validator = Validator::make(['id' => $id], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->profileRepository->findByIdOrFail($id);
    }

    public function notes($id)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:pr_profile,id'],
        ];
        $validator = Validator::make(['id' => $id], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }
        $profile = $this->profileRepository->getBasicProfile($id);
        $notes = [];
        if ($profile->customer_notes != null) {
            $notes[] = [
                'type' => 'customer_notes',
                'notes' => $profile->customer_notes,
                'modal' => 'profile',
                'name' => null,
            ];
        }
        if ($profile->employee_notes != null) {
            $notes[] = [
                'type' => 'employee_notes',
                'notes' => $profile->employee_notes,
                'modal' => 'profile',
                'name' => null,
            ];
        }
        foreach ($profile->accommodationsReservations as $note) {
            if ($note->customer_note != null) {
                $notes[] = [
                    'type' => 'customer_notes',
                    'notes' => $note->customer_notes,
                    'modal' => 'accommodations',
                    'name' => $note->accommodation->currentTranslation->name ?? $note->accommodation->name ?? null,
                ];
            }
            if ($note->hotel_note != null) {
                $notes[] = [
                    'type' => 'hotel_notes',
                    'notes' => $note->customer_notes,
                    'modal' => 'accommodations',
                    'name' => $note->accommodation->currentTranslation->name ?? $note->accommodation->name ?? null,
                ];
            }
        }

        // Collect notes from invoice services (excluding accommodation services)
        foreach ($profile->invoicesServices as $invoiceService) {
            if ($invoiceService->travel_tourism_type === 'accommodation') {
                continue;
            }

            if ($invoiceService->notes === null) {
                continue;
            }

            $serviceName = null;
            if ($invoiceService->service) {
                $serviceName = $invoiceService->service->currentTranslation->title
                    ?? $invoiceService->service->title
                    ?? null;
            } elseif ($invoiceService->city || $invoiceService->cityTo) {
                $from = $invoiceService->city->name ?? null;
                $to = $invoiceService->cityTo->name ?? null;
                $serviceName = trim(($from ? $from : '').($to ? ' -> '.$to : '')) ?: null;
            }

            $notes[] = [
                'type' => 'customer_notes',
                'notes' => $invoiceService->notes,
                'modal' => $invoiceService->travel_tourism_type,
                'name' => $serviceName ?? $invoiceService->travel_tourism_type,
            ];
        }

        return $notes;
    }

    public function create($request)
    {
        $rules = [
            // 'profile_number' => ['required', 'string', 'max:255', 'unique:pr_profile,profile_number'],
            'company_id' => ['nullable', 'integer', 'exists:companies,id'],
            'user_id' => ['nullable', 'integer', 'exists:users,id'],
            'customer_id' => ['nullable', 'integer', 'exists:pr_customer,id'],
            'arrival_airline' => ['nullable', 'integer', 'exists:airports,id'],
            'arrival_flight_number' => ['nullable', 'string', 'max:50'],
            'departure_airline' => ['nullable', 'integer', 'exists:airports,id'],
            'departure_flight_number' => ['nullable', 'string', 'max:50'],
            'arrival_date' => ['required', 'date', 'before:departure_date'],
            'arrival_time' => ['nullable', 'date_format:H:i'],
            'departure_date' => ['required', 'date', 'after:arrival_date'],
            'departure_time' => ['nullable', 'date_format:H:i'],
            // 'status' => ['required', 'in:done,followup,pending,cancel'],
            'active' => ['required', 'boolean'],
            'customer_notes' => ['nullable', 'string', 'max:1000'],
            'employee_notes' => ['nullable', 'string', 'max:1000'],
            'children_count' => ['required', 'integer', 'min:0'],
            'children_no_fee_count' => ['nullable', 'integer', 'min:0'],
            'adults_count' => ['required', 'integer', 'min:0'],
        ];

        $validator = Validator::make($request, $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }
        $data = $this->profileNumber($request['arrival_date']);
        $request = array_merge($request, $data);
        // $request['profile_number'] =$data['profile_number'];
        // $request['serial_number'] =$data['serial_number'];

        $profile = $this->profileRepository->create($request);

        // Create daily programs from arrival_date to departure_date inclusive
        $this->syncDailyProgramsForProfile($profile->id, $profile->arrival_date, $profile->departure_date);

        // Note: Profile notifications are now handled by ProfileObserver
        // which sends notifications to ADMIN group automatically on CRUD operations

        return $profile;
    }

    public function update($request)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:pr_profile,id'],
            //    'profile_number' => ['sometimes', 'string', 'max:255', 'unique:pr_profile,profile_number,' . $request['id']],
            'company_id' => ['sometimes', 'integer', 'exists:companies,id'],
            'customer_id' => ['sometimes', 'integer', 'exists:pr_customer,id'],
            'arrival_airline' => ['sometimes', 'nullable', 'integer', 'exists:airports,id'],
            'arrival_flight_number' => ['sometimes', 'nullable', 'string', 'max:50'],
            'departure_airline' => ['sometimes', 'nullable', 'integer', 'exists:airports,id'],
            'departure_flight_number' => ['sometimes', 'nullable', 'string', 'max:50'],
            'arrival_date' => ['sometimes', 'date', 'before:departure_date'],
            'arrival_time' => ['sometimes', 'nullable', 'date_format:H:i'],
            'departure_date' => ['sometimes', 'date', 'after:arrival_date'],
            'departure_time' => ['sometimes', 'nullable', 'date_format:H:i'],
            'status' => ['sometimes', 'in:done,followup,pending,cancel'],
            'active' => ['sometimes', 'boolean'],
            'customer_notes' => ['sometimes', 'nullable', 'string', 'max:1000'],
            'employee_notes' => ['sometimes', 'nullable', 'string', 'max:1000'],
            'children_count' => ['sometimes', 'integer', 'min:0'],
            'children_no_fee_count' => ['nullable', 'integer', 'min:0'],
            'adults_count' => ['sometimes', 'integer', 'min:0'],
        ];

        $validator = Validator::make($request, $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }
        $item = $this->profileRepository->findByIdOrFail($request['id']);
        // Check if arrival_date is actually changed by comparing dates properly
        if (isset($request['arrival_date'])) {
            $newArrivalDate = Carbon::parse($request['arrival_date'])->format('Y-m-d');
            $oldArrivalDate = Carbon::parse($item->arrival_date)->format('Y-m-d');

            if ($newArrivalDate != $oldArrivalDate) {
                $data = $this->profileNumber($request['arrival_date']);
                $request = array_merge($request, $data);
            } else {
                // If arrival_date hasn't changed, ensure serial_number and profile_number are not updated
                unset($request['serial_number']);
                unset($request['profile_number']);
            }
        }

        // Ensure provided profile_number (if any) is unique as well
        if (isset($request['profile_number'])) {
            $request['profile_number'] = $this->makeUniqueProfileNumber($request['profile_number'], (int) $request['id']);
        }

        if (isset($request['status']) && ! $this->checkStatus($request['id'], $request['status'])) {
            $validator->errors()->add('status', 'لا يمكن تغير الحاله طلما البروفيل لديه فاتورة');
            $this->setError($validator->errors());

            return false;
        }
        $updated = $this->profileRepository->update($item, $request);

        // Sync daily programs only if dates provided and changed
        $newArrival = $request['arrival_date'] ?? $item->arrival_date;
        $newDeparture = $request['departure_date'] ?? $item->departure_date;

        $arrivalChanged = array_key_exists('arrival_date', $request) && $newArrival != $item->arrival_date;
        $departureChanged = array_key_exists('departure_date', $request) && $newDeparture != $item->departure_date;

        if ($arrivalChanged || $departureChanged) {
            $this->syncDailyProgramsForProfile($updated->id, $newArrival, $newDeparture);
        }

        return $updated;
    }

    public function profileNumber($arrival)
    {

        $date = Carbon::parse($arrival);
        $autoIncrement = $this->profileRepository->getByTravaliDate($date->year, $date->month);
        $base = 'ETB-'.$date->month.'-'.$date->day.'-'.$autoIncrement;
        $unique = $this->makeUniqueProfileNumber($base);

        return ['profile_number' => $unique, 'serial_number' => $autoIncrement];
    }

    /**
     * Ensure profile_number is unique; when exists, append -2, -3, ... until unique.
     * Optionally pass $ignoreId to exclude current record during update.
     */
    private function makeUniqueProfileNumber(string $base, int $ignoreId = 0): string
    {
        $candidate = $base;
        $suffix = 2;
        while (\App\Models\Profile\Profile::query()
            ->when($ignoreId > 0, fn ($q) => $q->where('id', '!=', $ignoreId))
            ->where('profile_number', $candidate)
            ->exists()) {
            $candidate = $base.'-'.$suffix;
            $suffix++;
        }

        return $candidate;
    }

    private function syncDailyProgramsForProfile(int $profileId, $arrivalDate, $departureDate): void
    {
        $start = Carbon::parse($arrivalDate)->startOfDay();
        $end = Carbon::parse($departureDate)->startOfDay();

        $days = $start->diffInDays($end) + 1; // inclusive

        // Fetch existing programs (including soft-deleted to restore if needed)
        $existing = \App\Models\Profile\DailyPrograms::withTrashed()
            ->where('profile_id', $profileId)
            ->get()
            ->keyBy('day_number');

        // Build target set
        $current = $start->copy();
        for ($dayNumber = 1; $dayNumber <= $days; $dayNumber++) {
            $dayDate = $current->toDateString();
            if (isset($existing[$dayNumber])) {
                $program = $existing[$dayNumber];
                // Restore if soft-deleted
                if ($program->trashed()) {
                    $program->restore();
                }
                // Update date only if different and provided via recompute
                if ($program->day_date !== $dayDate) {
                    $program->day_date = $dayDate;
                    $program->save();
                }
            } else {
                // Create missing day
                \App\Models\Profile\DailyPrograms::create([
                    'profile_id' => $profileId,
                    'day_number' => $dayNumber,
                    'day_date' => $dayDate,
                ]);
            }

            $current->addDay();
        }

        // Soft delete extra days beyond new range
        foreach ($existing as $dayNumber => $program) {
            if ($dayNumber > $days && ! $program->trashed()) {
                $program->delete();
            }
        }
    }

    public function del(array $ids)
    {
        $rules = [
            'ids' => ['required', 'array', 'min:1'],
            'ids.*' => ['required', 'integer', 'exists:pr_profile,id'],
        ];
        $validator = Validator::make(['ids' => $ids], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->profileRepository->del($ids);
    }

    public function getActive()
    {
        return $this->profileRepository->getActive();
    }

    public function getAllChanges($id)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:pr_profile,id'],
        ];
        $validator = Validator::make(['id' => $id], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        $profile = $this->getById($id);
        if (! $profile) {
            return false;
        }

        return $this->profileRepository->getAllChanges($profile);
    }

    public function getColumnChanges($id, $columnName)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:pr_profile,id'],
            'columnName' => ['required', 'string', 'max:255'],
        ];
        $validator = Validator::make(['id' => $id, 'columnName' => $columnName], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        $profile = $this->getById($id);
        if (! $profile) {
            return false;
        }

        return $this->profileRepository->getColumnChanges($profile, $columnName);
    }

    public function checkReservations($id)
    {
        $services = ['tour_guide', 'tour_reps', 'daily_transportation', 'dining_entertainment', 'attractions_temples'];

        return [
            'hotel' => $this->accmodationRepository->checkReservation('hotel', $id),
            'cruise' => $this->accmodationRepository->checkReservation('cruise', $id),
            'service' => $this->invoiceServiceRepository->CheckInvoiceServices($services, $id),
            'single_transportation' => $this->invoiceServiceRepository->CheckInvoiceServices(['single_transportation'], $id),
            'additional_services' => $this->invoiceServiceRepository->CheckInvoiceServices(['additional_services'], $id),
        ];
    }

    public function getProfileShare($code)
    {
        $profile = $this->profileRepository->getByCode($code);
        if (! empty($profile)) {
            // تفاصيل الحجوزات
            // hotel with rooms
            $hotel = $this->accmodationRepository->getReservationsByProfile('hotel', $profile->id);
            $cruise = $this->accmodationRepository->getReservationsByProfile('cruise', $profile->id);
            $serives = $this->invoiceServiceRepository->getServicesForProfile($profile->id);
            $other = $this->invoiceServiceRepository->getOtherSerForProfile('additional_services', $profile->id);
            $single = $this->invoiceServiceRepository->getOtherSerForProfile('single_transportation', $profile->id);
            // معلومات الجنسية
            $nationality = null;
            if ($profile->customer && $profile->customer->nationality) {
                $nationality = [
                    'id' => $profile->customer->nationality->id,
                    'name' => $profile->customer->nationality->relationLoaded('currentTranslation')
                        ? ($profile->customer->nationality->currentTranslation->name ?? $profile->customer->nationality->name)
                        : $profile->customer->nationality->name,
                ];
            }
            // تفاصيل الدفع
            $cost = collect($profile->travelers)
                ->groupBy('currency_id')
                ->map(function ($group) {
                    $currency = $group->first()['currency'];

                    return [
                        'currency_id' => $currency->id,
                        'currency' => $currency['current_translation']['name'] ?? $currency['name'],
                        'total' => floatval($group->sum(fn ($item) => (float) $item['total_price'])),
                    ];
                })->values();
            // المدفوع
            $unpost = new UnpostedCollectionsRepository;
            $collector = $unpost->getPostedByCurrnecy($profile->id);
            $currency = $cost->pluck('currency_id')->toArray();
            $summary = [];
            foreach ($currency as $value) {
                $costBycurrency = collect($cost)->where('currency_id', $value)->first();
                $collectorBycurrency = collect($collector)->where('currency_id', $value)->first();
                $summary[] = [
                    'currency_id' => $value,
                    'currency_name' => $costBycurrency['currency'] ?? null,
                    'cost' => $costBycurrency['total'] ?? 0,
                    'collector' => $collectorBycurrency->total_amount ?? 0,
                    'remaining' => ($costBycurrency['total'] ?? 0) - ($collectorBycurrency->total_amount ?? 0),
                ];
            }

            return [
                'profile' => $profile,
                'hotel' => $hotel,
                'cruise' => $cruise,
                'single_transportation' => $single,
                'additional_services' => $other,
                'service' => $serives,
                'summary' => $summary,
                'nationality' => $nationality,
            ];

        }
        $this->setError(['code' => 'not Found']);

        return false;
    }

    public function checkStatus($profile, $status)
    {
        $myprofile = $this->profileRepository->getProfielStatus($profile);
        if ($status == 'cancel' && in_array($myprofile, ['followup', 'done'])) {
            return false;
        }

        return true;
    }
}
