<?php

namespace App\Http\Controllers\General;

use App\Http\Controllers\Controller;
use App\Http\Resources\General\OperationDataHistoryResource;
use App\Http\Resources\OperationDataCollection;
use App\Http\Resources\OperationDataResource;
use App\sys\ApiResponse;
use App\sys\Services\General\OperationDataService;
use Illuminate\Http\Request;

class OperationDataController extends Controller
{
    use ApiResponse;

    protected OperationDataService $operationDataService;

    private $operationDataTranslationService;

    public function __construct(OperationDataService $operationDataService)
    {
        $this->operationDataService = $operationDataService;
        $this->operationDataTranslationService = new \App\sys\Services\General\OperationDataTranslationService;
    }

    public function index()
    {
        $items = $this->operationDataService->getPaginatedOperationData();

        return $this->apiResponse(200, 'success', null, new OperationDataCollection($items));
    }

    public function show(int $id)
    {
        $item = $this->operationDataService->getOperationDataById($id);
        if (! $item) {
            $errors = $this->operationDataService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, new OperationDataResource($item));
    }

    public function store(Request $request)
    {
        $response = $this->operationDataService->createOperationData($request->all());
        if (! $response) {
            $errors = $this->operationDataService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, new OperationDataResource($response));
    }

    public function update(Request $request)
    {
        $response = $this->operationDataService->updateOperationData($request->all());
        if (! $response) {
            $errors = $this->operationDataService->errors();

            return $this->apiResponse(400, 'fail', null, $errors);
        }

        return $this->apiResponse(200, 'success', null, new OperationDataResource($response));
    }

    public function destroy(Request $request)
    {
        $ids = $request->input('ids', []);
        if (! is_array($ids) || empty($ids)) {
            return $this->apiResponse(400, 'No IDs provided for deletion', null, null);
        }
        $result = $this->operationDataService->del($ids);
        if ($result) {
            return $this->apiResponse(200, 'OperationData deleted successfully', null, null);
        }
        $errors = $this->operationDataService->errors();

        return $this->apiResponse(400, 'fail', $errors, null);
    }

    public function getActive()
    {
        $data = $this->operationDataService->getActive();
        $collect = new \App\Http\Resources\OperationDataCollection($data);

        return $this->apiResponse(200, 'Active OperationData retrieved successfully', null, $collect);
    }

    public function saveTranslation(Request $request, $id)
    {
        if (! $this->operationDataTranslationService->save($request->all(), $id)) {
            $errors = $this->operationDataTranslationService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, null);
    }

    public function getTranslation($id)
    {
        if ($data = $this->operationDataService->getByIdWithTranslation($id)) {
            $collection = new \App\Http\Resources\General\OperationData\OperationDataLangResource($data);

            return $this->apiResponse(200, 'success', null, $collection);
        }
        $errors = $this->operationDataService->errors();

        return $this->apiResponse(400, 'fail', $errors, null);
    }

    public function delLang($id)
    {
        if ($this->operationDataTranslationService->del($id)) {
            return $this->apiResponse(200, 'success', null, null);
        }

        $errors = $this->operationDataTranslationService->errors();

        return $this->apiResponse(400, 'fail', $errors, null);
    }

    public function getAllChanges($id)
    {
        $item = $this->operationDataService->getOperationDataById($id);
        if (! $item) {
            $errors = $this->operationDataService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }
        $changes = $this->operationDataService->getAllChanges($item);

        return $this->apiResponse(200, 'success', null, OperationDataHistoryResource::collection($changes));
    }
}
