<?php

namespace App\Http\Controllers\General;

use App\Http\Controllers\Controller;
use App\Http\Resources\General\Service\ServiceLangResource;
use App\Http\Resources\ServiceResource;
use App\Http\Resources\ServicesCollection;
use App\sys\ApiResponse;
use App\sys\Helper;
use App\sys\Services\General\ServiceService;
use App\sys\Services\General\ServiceTranslationService;
use Illuminate\Http\Request;

class ServiceController extends Controller
{
    use ApiResponse;

    protected ServiceService $serviceService;

    protected ServiceTranslationService $serviceTranslationService;

    public function __construct(ServiceService $serviceService, ServiceTranslationService $serviceTranslationService)
    {
        $this->serviceService = $serviceService;
        $this->serviceTranslationService = $serviceTranslationService;
    }

    public function index()
    {
        $services = $this->serviceService->getPaginatedCountries();

        return $this->apiResponse(200, 'success', null, new ServicesCollection($services));
    }

    public function getCurrencyAndTreeAccount()
    {
        $data = Helper::getTreeByCurrency();

        return $this->apiResponse(200, 'success', null, $data);
    }

    public function show(int $id)
    {
        if ($service = $this->serviceService->getServiceById($id)) {
            return $this->apiResponse(200, 'success', null, new ServiceResource($service));
        }
        $errors = $this->serviceService->errors();

        return $this->apiResponse(400, 'fail', $errors, null);
    }

    public function store(Request $request)
    {
        if ($response = $this->serviceService->createService($request->all())) {
            return $this->apiResponse(200, 'success', null, new ServiceResource($response));
        }
        $errors = $this->serviceService->errors();

        return $this->apiResponse(400, 'fail', $errors, null);
    }

    public function update(Request $request)
    {
        if ($response = $this->serviceService->updateService($request->all())) {
            return $this->apiResponse(200, 'success', null, new ServiceResource($response));
        }
        $errors = $this->serviceService->errors();

        return $this->apiResponse(400, 'fail', $errors, null);
    }

    public function getCurrencyWithTree(Request $request)
    {
        $currencyId = $request->get('currency_id');
        $data = $this->serviceService->getCurrencyWithTree($currencyId);

        return $this->apiResponse(200, 'success', null, $data);
    }

    public function getActive()
    {
        $services = $this->serviceService->getActive();

        return $this->apiResponse(200, 'success', null, new ServicesCollection($services));
    }

    public function getByType(string $type)
    {
        if ($services = $this->serviceService->getByType($type)) {
            return $this->apiResponse(200, 'success', null, new ServicesCollection($services));
        }
        $errors = $this->serviceService->errors();

        return $this->apiResponse(400, 'fail', $errors, null);
    }

    public function destroy(Request $request)
    {
        $ids = $request->input('ids', []);
        if (! is_array($ids) || empty($ids)) {
            return $this->apiResponse(400, 'No IDs provided for deletion', null, null);
        }
        $result = $this->serviceService->del($ids);
        if ($result) {
            return $this->apiResponse(200, 'Service deleted successfully', null, null);
        }
        $errors = $this->serviceService->errors();

        return $this->apiResponse(400, 'fail', $errors, null);
    }

    public function saveTranslation(Request $request, $id)
    {
        if (! $this->serviceTranslationService->save($request->all(), $id)) {
            $errors = $this->serviceTranslationService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, null);
    }

    public function getTranslation($id)
    {
        $data = $this->serviceService->getByIdWithTranslation($id);
        if ($data) {
            return $this->apiResponse(200, 'success', null, new ServiceLangResource($data));
        }
        $errors = $this->serviceService->errors();

        return $this->apiResponse(400, 'fail', $errors, null);
    }

    public function delLang($id)
    {
        if ($this->serviceTranslationService->del($id)) {
            return $this->apiResponse(200, 'success', null, null);
        }
        $errors = $this->serviceTranslationService->errors();

        return $this->apiResponse(400, 'fail', $errors, null);
    }

    public function getPricesByServiceAndCurrency($serviceId, $currencyId)
    {
        $data = $this->serviceService->getServicePrice($serviceId, $currencyId);

        return $this->apiResponse(200, 'success', null, $data);
    }
}
