<?php

namespace App\Observers;

use App\Models\Profile\AccommodationReservation;
use App\Models\Suppliers\Suppliers;
use App\sys\Enums\GroupEnum;
use App\sys\Services\Notification\NotificationService;

class AccommodationReservationObserver
{
    protected NotificationService $notificationService;

    /**
     * المجموعات المستهدفة بإشعارات حجز الإقامة
     */
    protected const TARGET_GROUPS = [
        GroupEnum::ADMIN,
        GroupEnum::ACCOUNTANT_MANAGER,
        GroupEnum::SALES_MANAGER,
        GroupEnum::RESERVATION_MANAGER,
    ];

    public function __construct(NotificationService $notificationService)
    {
        $this->notificationService = $notificationService;
    }

    /**
     * Handle the AccommodationReservation "created" event.
     */
    public function created(AccommodationReservation $reservation): void
    {
        $profileNumber = $reservation->profile?->profile_number;
        $accommodationName = $reservation->accommodation?->currentTranslation?->name
            ?? $reservation->accommodation?->name
            ?? 'غير محدد';

        $title = 'تم إنشاء حجز إقامة جديد';

        $messageParts = [];
        $messageParts[] = "تم إنشاء حجز إقامة جديد برقم {$reservation->id}";

        if ($profileNumber) {
            $messageParts[] = "لـ Profile رقم {$profileNumber}";
        }

        if ($accommodationName) {
            $messageParts[] = "في فندق/إقامة {$accommodationName}";
        }

        $message = implode(' - ', $messageParts);

        $data = [
            'reservation_id' => $reservation->id,
            'profile_id' => $reservation->profile_id,
            'accommodation_id' => $reservation->accommodation_id,
            'status' => $reservation->status,
            'type' => $reservation->type,
            'confirmation_date' => $reservation->confirmation_date,
            'confirmation_num' => $reservation->confirmation_num,
            'supplier_id' => $reservation->supplier_id,
            'city_id' => $reservation->city_id,
            'country_id' => $reservation->country_id,
            'action' => 'created',
        ];

        foreach (self::TARGET_GROUPS as $group) {
            $this->notificationService->sendToGroup(
                groupId: $group->value,
                type: 'accommodation_reservation',
                title: $title,
                message: $message,
                modelName: 'AccommodationReservation',
                modelId: $reservation->id,
                data: $data,
                pushChannel: 'group'
            );
        }

        // إرسال إشعار للـ Supplier إذا كان موجود
        if ($reservation->supplier_id) {
            $supplierName = $this->getSupplierName($reservation->supplier_id);

            $this->sendToManagementGroups(
                reservation: $reservation,
                title: 'تم تحديد Supplier لحجز إقامة',
                message: "تم تحديد Supplier {$supplierName} لحجز إقامة رقم {$reservation->id}",
                data: [
                    'supplier_id' => $reservation->supplier_id,
                    'supplier_name' => $supplierName,
                    'reservation_id' => $reservation->id,
                    'profile_id' => $reservation->profile_id,
                    'action' => 'supplier_set',
                ]
            );
        }
    }

    /**
     * Handle the AccommodationReservation "updated" event.
     */
    public function updated(AccommodationReservation $reservation): void
    {
        // التحقق من تغيير supplier_id
        if ($reservation->wasChanged('supplier_id')) {
            $oldSupplierId = $reservation->getOriginal('supplier_id');
            $newSupplierId = $reservation->supplier_id;
            $supplierName = $this->getSupplierName($newSupplierId);
            $oldSupplierName = $this->getSupplierName($oldSupplierId);

            // تحديد نوع الرسالة بناءً على القيمة القديمة والجديدة
            if (empty($oldSupplierId) && ! empty($newSupplierId)) {
                $title = 'تم تحديد Supplier لحجز إقامة';
                $message = "تم تحديد Supplier {$supplierName} لحجز إقامة رقم {$reservation->id}";
                $action = 'supplier_set';
            } else {
                $title = 'تم تغيير الـ Supplier لحجز إقامة';
                $message = "تم تغيير الـ Supplier لحجز إقامة رقم {$reservation->id} من {$oldSupplierName} إلى {$supplierName}";
                $action = 'supplier_changed';
            }

            // إرسال إشعار للجروبات الأربعة
            $this->sendToManagementGroups(
                reservation: $reservation,
                title: $title,
                message: $message,
                data: [
                    'old_supplier_id' => $oldSupplierId,
                    'new_supplier_id' => $newSupplierId,
                    'supplier_name' => $supplierName,
                    'old_supplier_name' => $oldSupplierName,
                    'action' => $action,
                ]
            );
        }
    }

    /**
     * إرسال إشعار للجروبات المستهدفة
     */
    protected function sendToManagementGroups(
        AccommodationReservation $reservation,
        string $title,
        string $message,
        array $data = []
    ): void {
        $baseData = [
            'reservation_id' => $reservation->id,
            'profile_id' => $reservation->profile_id,
            'accommodation_id' => $reservation->accommodation_id,
            'status' => $reservation->status,
            'type' => $reservation->type,
        ];

        $dataSend = array_merge($baseData, $data);

        foreach (self::TARGET_GROUPS as $group) {
            $this->notificationService->sendToGroup(
                groupId: $group->value,
                type: 'accommodation_reservation',
                title: $title,
                message: $message,
                modelName: 'AccommodationReservation',
                modelId: $reservation->id,
                data: $dataSend,
                pushChannel: 'group'
            );
        }
    }

    /**
     * جلب اسم الـ Supplier
     */
    protected function getSupplierName(?int $supplierId): string
    {
        if (! $supplierId) {
            return 'غير محدد';
        }

        $supplier = Suppliers::with('currentTranslation')->find($supplierId);

        if (! $supplier) {
            return 'غير محدد';
        }

        return $supplier->currentTranslation?->supplier_name
            ?? $supplier->supplier_name
            ?? 'غير محدد';
    }
}
