<?php

namespace App\Observers;

use App\Models\General\Currency;
use App\Models\Profile\AccommodationReservationRoom;
use App\sys\Enums\GroupEnum;
use App\sys\Services\Notification\NotificationService;

class AccommodationReservationRoomObserver
{
    protected NotificationService $notificationService;

    /**
     * المجموعات المستهدفة بإشعارات غرف حجز الإقامة
     */
    protected const TARGET_GROUPS = [
        GroupEnum::ADMIN,
        GroupEnum::ACCOUNTANT_MANAGER,
        GroupEnum::SALES_MANAGER,
        GroupEnum::RESERVATION_MANAGER,
    ];

    public function __construct(NotificationService $notificationService)
    {
        $this->notificationService = $notificationService;
    }

    /**
     * Handle the AccommodationReservationRoom "created" event.
     */
    public function created(AccommodationReservationRoom $room): void
    {
        $reservation = $room->reservation;
        $profile = $room->profile;

        $profileNumber = $profile?->profile_number ?? null;
        $roomName = $room->room_name ?? ($room->room_number ? "غرفة رقم {$room->room_number}" : "غرفة رقم {$room->id}");

        $title = 'تم إضافة غرفة جديدة لحجز إقامة';

        $messageParts = [];
        $messageParts[] = "تم إضافة {$roomName} للحجز رقم {$room->reservation_id}";

        if ($profileNumber) {
            $messageParts[] = "لـ Profile رقم {$profileNumber}";
        }

        $message = implode(' - ', $messageParts);

        $data = [
            'room_id' => $room->id,
            'reservation_id' => $room->reservation_id ?? null,
            'profile_id' => $room->profile_id ?? null,
            'check_in' => $room->check_in ?? null,
            'check_out' => $room->check_out ?? null,
            'nights_count' => $room->nights_count ?? 0,
            'adult_count' => $room->adult_count ?? 0,
            'childreen_count' => $room->childreen_count ?? 0,
            'no_fee_children_count' => $room->no_fee_children_count ?? 0,
            'room_number' => $room->room_number ?? null,
            'total_amount' => $room->total_amount ?? 0,
            'status' => $room->status ?? 'pending',
            'action' => 'created',
        ];

        foreach (self::TARGET_GROUPS as $group) {
            $this->notificationService->sendToGroup(
                groupId: $group->value,
                type: 'reservation_room',
                title: $title,
                message: $message,
                modelName: 'AccommodationReservationRoom',
                modelId: $room->id,
                data: $data,
                pushChannel: 'group'
            );
        }
    }

    /**
     * Handle the AccommodationReservationRoom "updated" event.
     */
    public function updated(AccommodationReservationRoom $room): void
    {
        $reservation = $room->reservation;
        $profile = $room->profile;
        $profileNumber = $profile?->profile_number ?? null;
        $roomName = $room->room_name ?? ($room->room_number ? "غرفة رقم {$room->room_number}" : "غرفة رقم {$room->id}");

        // التحقق من تغيير total_amount
        if ($room->wasChanged('total_amount')) {
            $oldTotalAmount = (float) ($room->getOriginal('total_amount') ?? 0);
            $newTotalAmount = (float) ($room->total_amount ?? 0);

            // تحديد نوع الرسالة بناءً على القيمة القديمة والجديدة
            if ($oldTotalAmount == 0 && $newTotalAmount > 0) {
                $title = 'تم تحديد سعر الغرفة';
                $message = "تم تحديد سعر {$roomName} في حجز إقامة رقم {$room->reservation_id} - السعر: {$newTotalAmount}";
                $action = 'price_set';
            } else {
                $title = 'تم تغيير سعر الغرفة';
                $message = "تم تغيير سعر {$roomName} في حجز إقامة رقم {$room->reservation_id} من {$oldTotalAmount} إلى {$newTotalAmount}";
                $action = 'price_changed';
            }

            if ($profileNumber) {
                $message .= " - Profile رقم {$profileNumber}";
            }

            $this->sendToManagementGroups(
                room: $room,
                title: $title,
                message: $message,
                data: [
                    'old_total_amount' => $oldTotalAmount,
                    'new_total_amount' => $newTotalAmount,
                    'action' => $action,
                ]
            );
        }

        // التحقق من تغيير currency_id
        if ($room->wasChanged('currency_id')) {
            $oldCurrencyId = $room->getOriginal('currency_id');
            $newCurrencyId = $room->currency_id;
            $currencyName = $this->getCurrencyName($newCurrencyId);

            // تحديد نوع الرسالة بناءً على القيمة القديمة والجديدة
            if (empty($oldCurrencyId) && ! empty($newCurrencyId)) {
                $title = 'تم تحديد عملة الغرفة';
                $message = "تم تحديد عملة {$roomName} في حجز إقامة رقم {$room->reservation_id} - العملة: {$currencyName}";
                $action = 'currency_set';
            } else {
                $oldCurrencyName = $this->getCurrencyName($oldCurrencyId);
                $title = 'تم تغيير عملة الغرفة';
                $message = "تم تغيير عملة {$roomName} في حجز إقامة رقم {$room->reservation_id} من {$oldCurrencyName} إلى {$currencyName}";
                $action = 'currency_changed';
            }

            if ($profileNumber) {
                $message .= " - Profile رقم {$profileNumber}";
            }

            $this->sendToManagementGroups(
                room: $room,
                title: $title,
                message: $message,
                data: [
                    'old_currency_id' => $oldCurrencyId,
                    'new_currency_id' => $newCurrencyId,
                    'currency_name' => $currencyName,
                    'old_currency_name' => $oldCurrencyName ?? 'غير محدد',
                    'action' => $action,
                ]
            );
        }

        // التحقق من تغيير status
        if ($room->wasChanged('status')) {
            $oldStatus = $room->getOriginal('status') ?? 'غير محدد';
            $newStatus = $room->status ?? 'غير محدد';

            $statusLabels = [
                'pending' => 'قيد الانتظار',
                'confirmed' => 'مؤكد',
                'cancelled' => 'ملغي',
                'done' => 'منجز',
            ];

            $oldStatusLabel = $statusLabels[$oldStatus] ?? $oldStatus;
            $newStatusLabel = $statusLabels[$newStatus] ?? $newStatus;

            $title = 'تم تغيير حالة الغرفة';
            $message = "تم تغيير حالة {$roomName} في حجز إقامة رقم {$room->reservation_id} من {$oldStatusLabel} إلى {$newStatusLabel}";

            if ($profileNumber) {
                $message .= " - Profile رقم {$profileNumber}";
            }

            $this->sendToManagementGroups(
                room: $room,
                title: $title,
                message: $message,
                data: [
                    'old_status' => $oldStatus,
                    'new_status' => $newStatus,
                    'old_status_label' => $oldStatusLabel,
                    'new_status_label' => $newStatusLabel,
                    'action' => 'status_changed',
                ]
            );
        }
    }

    /**
     * إرسال إشعار للجروبات المستهدفة
     */
    protected function sendToManagementGroups(
        AccommodationReservationRoom $room,
        string $title,
        string $message,
        array $data = []
    ): void {
        $roomName = $room->room_number ? "غرفة رقم {$room->room_number}" : "غرفة رقم {$room->id}";

        $baseData = [
            'room_id' => $room->id,
            'reservation_id' => $room->reservation_id ?? null,
            'profile_id' => $room->profile_id ?? null,
            'room_name' => $room->room_name ?? ($room->room_number ? "غرفة رقم {$room->room_number}" : "غرفة رقم {$room->id}"),
            'room_number' => $room->room_number ?? null,
            'check_in' => $room->check_in ?? null,
            'check_out' => $room->check_out ?? null,
            'total_amount' => $room->total_amount ?? 0,
            'currency_id' => $room->currency_id ?? null,
            'status' => $room->status ?? 'pending',
        ];

        $dataSend = array_merge($baseData, $data);

        foreach (self::TARGET_GROUPS as $group) {
            $this->notificationService->sendToGroup(
                groupId: $group->value,
                type: 'reservation_room',
                title: $title,
                message: $message,
                modelName: 'AccommodationReservationRoom',
                modelId: $room->id,
                data: $dataSend,
                pushChannel: 'group'
            );
        }
    }

    /**
     * جلب اسم العملة
     */
    protected function getCurrencyName(?int $currencyId): string
    {
        if (! $currencyId) {
            return 'غير محدد';
        }

        $currency = Currency::with('currentTranslation')->find($currencyId);

        if (! $currency) {
            return 'غير محدد';
        }

        return $currency->currentTranslation?->name
            ?? $currency->name
            ?? 'غير محدد';
    }
}
