<?php

namespace App\sys\Repository\General;

use App\Models\General\OperationData;

class OperationDataRepository
{
    private $columns = [
        'id' => 'id',
        'name' => 'name',
        'active' => 'is_active',
        'created_at' => 'created_at',
        'updated_at' => 'updated_at',
    ];

    public function getPaginated()
    {
        $column = request('sort_by', null);
        $order = request('sort_order', 'asc');
        $name = request('name', null);
        $limit = request('limit', 15);

        return OperationData::when($name, function ($q, $name) {
            $q->whereHas('currentTranslation', function ($q) use ($name) {
                $q->where('name', 'LIKE', "%$name%");
            });
        })->when($column && array_key_exists($column, $this->columns), function ($query) use ($column, $order) {
            if ($column == 'name') {
                $query->leftJoin('ge_operation_data_translation', function ($join) {
                    $join->on('ge_operation_data.id', '=', 'ge_operation_data_translation.operation_data_id')
                        ->where('ge_operation_data_translation.lang_id', app('lang_id'));
                })->orderBy('ge_operation_data_translation.name', $order)->select('ge_operation_data.*');
            } else {
                $query->orderBy($this->columns[$column], $order);
            }
        })->with('currentTranslation')->paginate($limit);
    }

    public function findByIdOrFail(int $id)
    {
        return OperationData::with('currentTranslation')->find($id);
    }

    public function create(array $data)
    {
        $item = new OperationData;
        $item->name = $data['name'];
        $item->is_active = $data['active'];
        $item->save();

        return $item;
    }

    public function update(OperationData $item, array $data)
    {
        $item->name = $data['name'] ?? $item->name;
        $item->is_active = $data['active'] ?? $item->is_active;
        $item->save();

        return $item;
    }

    public function delete(OperationData $item)
    {
        return $item->delete();
    }

    public function del(array $ids)
    {
        return OperationData::whereIn('id', $ids)->delete();
    }

    public function getActive()
    {
        return OperationData::with('currentTranslation')->where('is_active', 1)->get();
    }

    public function getByIdWithTranslation($id)
    {
        return OperationData::with('translations')->find($id);
    }

    public function getAllChanges($operationData)
    {
        return $operationData->audits()
            ->with(['user' => function ($query) {
                $query->select('id', 'name');
            }])
            ->get()
            ->map(function ($audit) {
                $oldValues = $audit->old_values ?? [];
                $newValues = $audit->new_values ?? [];

                return [
                    'audit_id' => $audit->id,
                    'user_id' => $audit->user_id ?? null,
                    'user' => $audit->user ? $audit->user->toArray() : null,
                    'old_values' => $oldValues,
                    'new_values' => $newValues,
                    'changed_at' => $audit->created_at,
                    'event' => $audit->event,
                    'ip_address' => $audit->ip_address,
                    'user_agent' => $audit->user_agent,
                ];
            })
            ->values();
    }
}
