<?php

namespace App\sys\Services\General;

use App\sys\Repository\General\CompanyTranslationRepository;
use App\sys\Services;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;

class CompanyTranslationService extends Services
{
    private $companyTranslation;

    public function __construct()
    {
        $this->companyTranslation = new CompanyTranslationRepository;
    }

    public function save(array $data, $company_id)
    {
        $rules = ['company_id' => 'required|integer|exists:companies,id'];
        $combinations = [];
        $customErrors = [];
        foreach ($data as $index => $row) {
            $rowId = $row['id'] ?? null;
            $languageId = $row['language_id'] ?? null;
            $rules["$index.id"] = 'sometimes|nullable|exists:company_translation,id';
            $rules["$index.language_id"] = [
                Rule::requiredIf(empty($rowId)),
                'exists:languages,id',
                Rule::unique('company_translation', 'lang_id')
                    ->where(fn ($query) => $query->where('company_id', $company_id))
                    ->ignore($rowId),
            ];
            $rules["$index.name"] = 'nullable|string|max:250';
            $rules["$index.companie_address"] = 'nullable|string|max:300';
            $rules["$index.default_invoice_footer"] = 'nullable|string';
            if ($languageId) {
                $key = $company_id.'-'.$languageId;
                if (in_array($key, $combinations)) {
                    $customErrors["$index.language_id"] = 'Same language duplicated for the same Company.';
                } else {
                    $combinations[] = $key;
                }
            }
        }
        $validator = Validator::make(array_merge($data, ['company_id' => $company_id]), $rules);
        $validator->after(function ($validator) use ($customErrors) {
            foreach ($customErrors as $field => $message) {
                $validator->errors()->add($field, $message);
            }
        });
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->companyTranslation->save($data, $company_id);
    }

    public function get($company_id)
    {
        return $this->companyTranslation->get($company_id);
    }

    public function del($id)
    {
        $rules = [
            'id' => 'required|integer|exists:company_translation,id',
        ];
        $validator = Validator::make(['id' => $id], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->companyTranslation->del($id);
    }
}
