<?php

namespace App\sys\Services\Profile;

use App\sys\Repository\Profile\CustomerRepository;
use App\sys\Services;
use Illuminate\Support\Facades\Validator;

class CustomerService extends Services
{
    protected CustomerRepository $customerRepository;

    public function __construct(CustomerRepository $customerRepository)
    {
        $this->customerRepository = $customerRepository;
    }

    public function getPaginated($filters = [])
    {
        return $this->customerRepository->getPaginated($filters);
    }

    public function getById($id)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:pr_customer,id'],
        ];
        $validator = Validator::make(['id' => $id], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->customerRepository->findByIdOrFail($id);
    }

    public function create($request)
    {
        $rules = [
            'full_name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'email', 'max:255', 'unique:pr_customer,email'],
            'phone' => ['nullable', 'string', 'max:50', 'unique:pr_customer,phone'],
            'nationality_id' => ['nullable', 'integer', 'exists:nationalities,id'],
            'country_id' => ['nullable', 'integer', 'exists:countries,id'],
            'passport' => ['nullable', 'string', 'max:50'],
            'company_id' => ['nullable', 'integer', 'exists:companies,id'],
            'user_id' => ['nullable', 'integer', 'exists:users,id'],
            'character_status' => ['nullable', 'string', 'in:active,inactive,vip,new,blocked'],
            'notes' => ['nullable', 'string', 'max:1000'],
            'type' => ['required', 'string', 'in:company,personal'],
            'active' => ['required', 'in:0,1'],
        ];
        $validator = Validator::make($request, $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->customerRepository->create($request);
    }

    public function update($request)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:pr_customer,id'],
            'full_name' => ['sometimes', 'string', 'max:255'],
            'email' => ['sometimes', 'nullable', 'string', 'email', 'max:255', 'unique:pr_customer,email,'.$request['id']],
            'phone' => ['sometimes', 'nullable', 'string', 'max:50', 'unique:pr_customer,phone,'.$request['id']],
            'nationality_id' => ['sometimes', 'nullable', 'integer', 'exists:nationalities,id'],
            'country_id' => ['sometimes', 'nullable', 'integer', 'exists:countries,id'],
            'passport' => ['sometimes', 'string', 'max:50'],
            'company_id' => ['sometimes', 'nullable', 'integer', 'exists:companies,id'],
            'user_id' => ['sometimes', 'nullable', 'integer', 'exists:users,id'],
            'character_status' => ['sometimes', 'nullable', 'string', 'in:active,inactive,vip,new,blocked'],
            'notes' => ['sometimes', 'nullable', 'string', 'max:1000'],
            'type' => ['sometimes', 'in:company,personal'],
            'active' => ['sometimes', 'in:0,1'],
        ];
        $validator = Validator::make($request, $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }
        $item = $this->customerRepository->findByIdOrFail($request['id']);

        return $this->customerRepository->update($item, $request);
    }

    public function delete($id)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:pr_customer,id'],
        ];
        $validator = Validator::make(['id' => $id], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }
        $item = $this->customerRepository->findByIdOrFail($id);

        return $this->customerRepository->delete($item);
    }

    public function del(array $ids)
    {
        $rules = [
            'ids' => ['required', 'array', 'min:1'],
            'ids.*' => ['required', 'integer', 'exists:pr_customer,id'],
        ];
        $validator = Validator::make(['ids' => $ids], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->customerRepository->del($ids);
    }

    public function getActive()
    {
        return $this->customerRepository->getActive();
    }

    public function getAllChanges($customer)
    {
        return $this->customerRepository->getAllChanges($customer);
    }
}
