<?php

namespace App\Http\Controllers\Accommodation;

use App\Http\Controllers\Controller;
use App\Http\Resources\Accommodation\Extra\ExtraLangResource;
use App\Http\Resources\Accommodation\ExtraHistoryResource;
use App\Http\Resources\ExtraCollection;
use App\Http\Resources\ExtraResource;
use App\sys\ApiResponse;
use App\sys\Services\Accommodation\ExtraService;
use App\sys\Services\Accommodation\ExtraTranslationServices;
use Illuminate\Http\Request;

class ExtraController extends Controller
{
    use ApiResponse;

    protected ExtraService $extraService;

    protected ExtraTranslationServices $extraTranslationServices;

    public function __construct(ExtraService $extraService, ExtraTranslationServices $extraTranslationServices)
    {
        $this->extraService = $extraService;
        $this->extraTranslationServices = $extraTranslationServices;
    }

    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $extras = $this->extraService->getPaginatedExtras();

        return $this->apiResponse(200, 'success', null, new ExtraCollection($extras));
    }

    public function getActive()
    {
        $data = $this->extraService->getActive();

        return $this->apiResponse(200, 'success', null, new ExtraCollection($data));
    }

    /**
     * Display the specified resource.
     */
    public function show(int $id)
    {
        $extra = $this->extraService->getExtraById($id);

        if (! $extra) {
            $errors = $this->extraService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, new ExtraResource($extra));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $response = $this->extraService->createExtra($request->all());

        if (! $response) {
            $errors = $this->extraService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, new ExtraResource($response));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request)
    {
        $response = $this->extraService->updateExtra($request->all());

        if (! $response) {
            $errors = $this->extraService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, new ExtraResource($response));
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Request $request)
    {
        $ids = $request->input('ids', []);
        if (! is_array($ids) || empty($ids)) {
            return $this->apiResponse(400, 'No IDs provided for deletion', null, null);
        }
        $result = $this->extraService->del($ids);
        if ($result) {
            return $this->apiResponse(200, 'Extras deleted successfully', null, null);
        }
        $errors = $this->extraService->errors();

        return $this->apiResponse(400, 'fail', $errors, null);
    }

    public function SaveTranslation(Request $request, $id)
    {
        if (! $this->extraTranslationServices->save($request->all(), $id)) {
            $errors = $this->extraTranslationServices->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, null);
    }

    public function getTranslation($id)
    {
        if ($data = $this->extraService->getByidWithTranslation($id)) {
            $collection = new ExtraLangResource($data);

            return $this->apiResponse(200, 'success', null, $collection);
        }
        $errors = $this->extraService->errors();

        return $this->apiResponse(400, 'fail', $errors, null);
    }

    public function delLang($id)
    {
        if ($this->extraTranslationServices->del($id)) {
            return $this->apiResponse(200, 'success', null, null);
        }

        $errors = $this->extraTranslationServices->errors();

        return $this->apiResponse(400, 'fail', $errors, null);
    }

    public function getAllChanges($id)
    {
        $extra = $this->extraService->getExtraById($id);
        if (! $extra) {
            $errors = $this->extraService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        $changes = $this->extraService->getAllChanges($extra);

        return $this->apiResponse(200, 'success', null, ExtraHistoryResource::collection($changes));
    }
}
