<?php

namespace App\Http\Controllers;

use App\Models\Airports;
use App\sys\ApiResponse;
use Illuminate\Support\Facades\DB;

class AirportsController extends Controller
{
    use ApiResponse;

    public function search($code = null)
    {
        $airports = Airports::leftJoin('countries', 'airports.country_id', '=', 'countries.id')
            ->leftJoin('country_translation as country_trans', function ($join) {
                $join->on('countries.id', '=', 'country_trans.country_id')
                    ->where('country_trans.lang_id', app('lang_id'));
            })
            ->leftJoin('cities', 'airports.city_id', '=', 'cities.id')
            ->leftJoin('city_translation as city_trans', function ($join) {
                $join->on('cities.id', '=', 'city_trans.city_id')
                    ->where('city_trans.lang_id', app('lang_id'));
            })
            ->when($code, function ($query, $code) {
                $query->where(function ($query) use ($code) {
                    $query->where('airports.code', 'like', "%$code%")
                        ->orWhere('airports.name_en', 'like', "%$code%")
                        ->orWhere('airports.name_ar', 'like', "%$code%")
                        ->orWhere('country_trans.name', 'like', "%$code%")
                        ->orWhere('countries.name', 'like', "%$code%")
                        ->orWhere('city_trans.name', 'like', "%$code%")
                        ->orWhere('cities.name', 'like', "%$code%");
                });
            })
            ->select(
                'airports.id',
                'airports.code',
                DB::raw('COALESCE(airports.name_ar, airports.name_en) as name'),
                DB::raw('COALESCE(country_trans.name, countries.name, "") as country_name'),
                DB::raw('COALESCE(city_trans.name, cities.name, "") as city_name'),
                DB::raw('CONCAT_WS(" - ", 
                    COALESCE(airports.name_ar, airports.name_en, ""), 
                    airports.code, 
                    COALESCE(country_trans.name, countries.name, ""), 
                    COALESCE(city_trans.name, cities.name, "")
                ) as display')
            )
            ->when($code, function ($query, $code) {
                // Priority: exact code matches first, then code starts with, then others
                $query->orderByRaw('
                CASE 
                    WHEN airports.code = ? THEN 1
                    WHEN airports.code LIKE ? THEN 2
                    ELSE 3
                END
            ', [$code, "$code%"])
                    ->orderBy('airports.code', 'asc');
            }, function ($query) {
                $query->orderBy('airports.code', 'asc');
            })
            ->groupBy('airports.id', 'airports.code', 'airports.name_ar', 'airports.name_en', 'country_trans.name', 'countries.name', 'city_trans.name', 'cities.name')
            ->paginate();

        return $this->apiResponse(200, 'airports', null, $airports);
    }
}
