<?php

namespace App\Http\Controllers\General;

use App\Http\Controllers\Controller;
use App\Http\Resources\CategoryCollection;
use App\Http\Resources\CategoryResource;
use App\Http\Resources\General\CategoryHistoryResource;
use App\sys\ApiResponse;
use App\sys\Services\General\CategoryService;
use App\sys\Services\General\CategoryTranslationService;
use Illuminate\Http\Request;

class CategoryController extends Controller
{
    use ApiResponse;

    protected $categoryService;

    protected $categoryTranslationService;

    public function __construct(CategoryService $categoryService)
    {
        $this->categoryService = $categoryService;
        $this->categoryTranslationService = new CategoryTranslationService;
    }

    public function index()
    {
        $categories = $this->categoryService->getPaginatedCategories();

        return $this->apiResponse(200, 'success', null, new CategoryCollection($categories));
    }

    public function show($id)
    {
        $category = $this->categoryService->getCategoryById($id);
        if (! $category) {
            $errors = $this->categoryService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, new CategoryResource($category));
    }

    public function create(Request $request)
    {
        $response = $this->categoryService->createCategory($request->all());
        if (! $response) {
            $errors = $this->categoryService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, new CategoryResource($response));
    }

    public function update(Request $request)
    {
        $response = $this->categoryService->updateCategory($request->all());
        if (! $response) {
            $errors = $this->categoryService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, new CategoryResource($response));
    }

    public function getActive()
    {
        $data = $this->categoryService->getActive();
        $collect = new CategoryCollection($data);

        return $this->apiResponse(200, 'Active Categories retrieved successfully', null, $collect);
    }

    public function destroy(Request $request)
    {
        $ids = $request->input('ids', []);
        if (! is_array($ids) || empty($ids)) {
            return $this->apiResponse(400, 'No IDs provided for deletion', null, null);
        }
        $result = $this->categoryService->del($ids);
        if ($result) {
            return $this->apiResponse(200, 'Categories deleted successfully', null, null);
        }
        $errors = $this->categoryService->errors();

        return $this->apiResponse(400, 'fail', $errors, null);
    }

    public function saveTranslation(Request $request, $id)
    {
        if (! $this->categoryTranslationService->save($request->all(), $id)) {
            $errors = $this->categoryTranslationService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, null);
    }

    public function getTranslation($id)
    {
        if ($data = $this->categoryService->getByIdWithTranslation($id)) {
            $collection = new \App\Http\Resources\General\Category\CategoryLangResource($data);

            return $this->apiResponse(200, 'success', null, $collection);
        }
        $errors = $this->categoryService->errors();

        return $this->apiResponse(400, 'fail', $errors, null);
    }

    public function delLang($id)
    {
        if ($this->categoryTranslationService->del($id)) {
            return $this->apiResponse(200, 'success', null, null);
        }
        $errors = $this->categoryTranslationService->errors();

        return $this->apiResponse(400, 'fail', $errors, null);
    }

    public function getAllChanges($id)
    {
        $category = $this->categoryService->getCategoryById($id);
        if (! $category) {
            $errors = $this->categoryService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        $changes = $this->categoryService->getAllChanges($category);

        return $this->apiResponse(200, 'success', null, CategoryHistoryResource::collection($changes));
    }
}
