<?php

namespace App\Http\Controllers\General;

use App\Http\Controllers\Controller;
use App\Http\Resources\General\RestaurantHistoryResource;
use App\Http\Resources\RestaurantCollection;
use App\Http\Resources\RestaurantResource;
use App\sys\ApiResponse;
use App\sys\Services\General\RestaurantService;
use Illuminate\Http\Request;

class RestaurantController extends Controller
{
    use ApiResponse;

    protected RestaurantService $restaurantService;

    private $restaurantTranslationService;

    public function __construct(RestaurantService $restaurantService)
    {
        $this->restaurantService = $restaurantService;
        $this->restaurantTranslationService = new \App\sys\Services\General\RestaurantTranslationService;
    }

    public function index()
    {
        $items = $this->restaurantService->getPaginatedRestaurant();

        return $this->apiResponse(200, 'success', null, new RestaurantCollection($items));
    }

    public function show(int $id)
    {
        $item = $this->restaurantService->getRestaurantById($id);
        if (! $item) {
            $errors = $this->restaurantService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, new RestaurantResource($item));
    }

    public function store(Request $request)
    {
        $response = $this->restaurantService->createRestaurant($request->all());
        if (! $response) {
            $errors = $this->restaurantService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, new RestaurantResource($response));
    }

    public function update(Request $request)
    {
        $response = $this->restaurantService->updateRestaurant($request->all());
        if (! $response) {
            $errors = $this->restaurantService->errors();

            return $this->apiResponse(400, 'fail', null, $errors);
        }

        return $this->apiResponse(200, 'success', null, new RestaurantResource($response));
    }

    public function destroy(Request $request)
    {
        $ids = $request->input('ids', []);
        if (! is_array($ids) || empty($ids)) {
            return $this->apiResponse(400, 'No IDs provided for deletion', null, null);
        }
        $result = $this->restaurantService->del($ids);
        if ($result) {
            return $this->apiResponse(200, 'Restaurant deleted successfully', null, null);
        }
        $errors = $this->restaurantService->errors();

        return $this->apiResponse(400, 'fail', $errors, null);
    }

    public function getActive()
    {
        $data = $this->restaurantService->getActive();
        $collect = new \App\Http\Resources\RestaurantCollection($data);

        return $this->apiResponse(200, 'Active Restaurant retrieved successfully', null, $collect);
    }

    public function saveTranslation(Request $request, $id)
    {
        if (! $this->restaurantTranslationService->save($request->all(), $id)) {
            $errors = $this->restaurantTranslationService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, null);
    }

    public function getTranslation($id)
    {
        if ($data = $this->restaurantService->getByIdWithTranslation($id)) {
            $collection = new \App\Http\Resources\General\Restaurant\RestaurantLangResource($data);

            return $this->apiResponse(200, 'success', null, $collection);
        }
        $errors = $this->restaurantService->errors();

        return $this->apiResponse(400, 'fail', $errors, null);
    }

    public function delLang($id)
    {
        if ($this->restaurantTranslationService->del($id)) {
            return $this->apiResponse(200, 'success', null, null);
        }

        $errors = $this->restaurantTranslationService->errors();

        return $this->apiResponse(400, 'fail', $errors, null);
    }

    public function getAllChanges($id)
    {
        $item = $this->restaurantService->getRestaurantById($id);
        if (! $item) {
            $errors = $this->restaurantService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }
        $changes = $this->restaurantService->getAllChanges($item);

        return $this->apiResponse(200, 'success', null, RestaurantHistoryResource::collection($changes));
    }
}
