<?php

namespace App\Http\Controllers\Invoice;

use App\Http\Controllers\Controller;
use App\Http\Resources\Invoice\InvoiceServicesCollection;
use App\Models\invoice\InvoiceServices;
use App\sys\ApiResponse;
use App\sys\Services\Invoice\AmadeusAirParserService;
use App\sys\Services\Invoice\InvoiceServicesService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

/**
 * Controller for handling Amadeus Interface Record (A.I.R.) file uploads
 * Processes A.I.R. files and creates/updates InvoiceServices records
 */
class AmadeusAirController extends Controller
{
    use ApiResponse;

    protected AmadeusAirParserService $parserService; // Service for parsing A.I.R. files

    protected InvoiceServicesService $invoiceServicesService; // Service for managing InvoiceServices

    /**
     * Constructor - inject dependencies
     *
     * @param  AmadeusAirParserService  $parserService  Parser service instance
     * @param  InvoiceServicesService  $invoiceServicesService  Invoice services service instance
     */
    public function __construct(
        AmadeusAirParserService $parserService,
        InvoiceServicesService $invoiceServicesService
    ) {
        $this->parserService = $parserService; // حفظ parser service
        $this->invoiceServicesService = $invoiceServicesService; // حفظ invoice services service
    }

    /**
     * Upload and parse A.I.R. file
     * Handles file upload, parsing, and creates/updates InvoiceServices based on record type
     *
     * @param  Request  $request  HTTP request containing file and profile_id
     * @return \Illuminate\Http\JsonResponse JSON response with success/error status
     */
    public function uploadAirFile(Request $request)
    {
        // التحقق من صحة البيانات المرسلة
        // المطلوب: ملف نصي (txt) و profile_id موجود في قاعدة البيانات
        $validator = Validator::make($request->all(), [
            'file' => ['required', 'file', 'mimes:txt'], // الملف مطلوب، يجب أن يكون ملف نصي
            // profile_id أصبح اختياري (sometimes) لدعم رفع الملف بدون بروفايل
            'profile_id' => ['sometimes', 'nullable', 'integer', 'exists:pr_profile,id'],
        ]);

        // إذا فشل التحقق، إرجاع الأخطاء
        if ($validator->fails()) {
            return $this->apiResponse(400, 'fail', $validator->errors(), null);
        }

        // قراءة محتوى الملف
        $file = $request->file('file'); // الحصول على الملف المرفوع
        $fileContent = file_get_contents($file->getRealPath()); // قراءة محتوى الملف
        $profileId = $request->filled('profile_id') ? (int) $request->input('profile_id') : null; // profile_id اختياري

        // تحليل ملف A.I.R. واستخراج البيانات
        $parsedData = $this->parserService->parseAirFile($fileContent, $profileId);

        // إذا فشل التحليل، إرجاع الأخطاء
        if ($parsedData === false) {
            $errors = $this->parserService->errors(); // الحصول على الأخطاء من parser service

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        // استخراج PNR ونوع السجل من البيانات المحللة
        $pnr = $parsedData['pnr_number']; // رقم PNR
        $recordType = $this->parserService->extractRecordType($fileContent); // نوع السجل (7A, MA, RF, NT, AMD)

        // حفظ الملف في storage للرجوع إليه لاحقاً
        $savedFilePath = $this->parserService->saveAirFile($fileContent, $pnr);

        // تحديد الإجراء بناءً على نوع السجل
        $action = 'created'; // الإجراء الافتراضي: إنشاء
        $invoiceService = null; // متغير لحفظ InvoiceService

        // معالجة أنواع السجلات المختلفة
        switch ($recordType) {
            case '7A': // Booking - حجز جديد
            case 'NT': // NDC Ticket - تذكرة NDC
                // إنشاء invoice service جديد
                $invoiceService = $this->invoiceServicesService->create($parsedData);

                // إذا فشل الإنشاء، إرجاع الأخطاء
                if (! $invoiceService) {
                    $errors = $this->invoiceServicesService->errors();

                    return $this->apiResponse(400, 'fail', $errors, null);
                }

                $action = 'created'; // الإجراء: تم الإنشاء
                break;

            case 'MA': // Void - إلغاء حجز
                // البحث عن invoice service موجود بنفس PNR
                $existing = InvoiceServices::where('pnr_number', $pnr)
                    ->where('travel_tourism_type', 'flights') // فقط خدمات الطيران
                    ->first();

                // إذا تم العثور على service موجود
                if ($existing) {
                    // تحديث حالة service إلى cancelled (ملغى)
                    $this->invoiceServicesService->updateStatus([
                        'id' => $existing->id, // معرف service
                        'status' => 'cancelled', // الحالة الجديدة: ملغى
                    ]);

                    $invoiceService = $existing->fresh(); // تحديث البيانات من قاعدة البيانات
                    $action = 'cancelled'; // الإجراء: تم الإلغاء
                } else {
                    // إذا لم يتم العثور على service، إرجاع خطأ
                    return $this->apiResponse(404, 'fail', ['pnr' => ['Invoice service with PNR '.$pnr.' not found']], null);
                }
                break;

            case 'RF': // Refund - استرداد
            case 'AMD': // Amendment - تعديل
                // البحث عن invoice service موجود بنفس PNR
                $existing = InvoiceServices::where('pnr_number', $pnr)
                    ->where('travel_tourism_type', 'flights') // فقط خدمات الطيران
                    ->first();

                // إذا تم العثور على service موجود
                if ($existing) {
                    // دمج البيانات المحللة مع معرف service الموجود للتحديث
                    $updateData = array_merge($parsedData, ['id' => $existing->id]);

                    // تحديث service الموجود
                    $invoiceService = $this->invoiceServicesService->update($updateData);

                    // إذا فشل التحديث، إرجاع الأخطاء
                    if (! $invoiceService) {
                        $errors = $this->invoiceServicesService->errors();

                        return $this->apiResponse(400, 'fail', $errors, null);
                    }

                    $action = 'updated'; // الإجراء: تم التحديث
                } else {
                    // إذا لم يتم العثور على service، إنشاء جديد
                    $invoiceService = $this->invoiceServicesService->create($parsedData);

                    // إذا فشل الإنشاء، إرجاع الأخطاء
                    if (! $invoiceService) {
                        $errors = $this->invoiceServicesService->errors();

                        return $this->apiResponse(400, 'fail', $errors, null);
                    }

                    $action = 'created'; // الإجراء: تم الإنشاء
                }
                break;

            default:
                // إذا كان نوع السجل غير مدعوم، إرجاع خطأ
                return $this->apiResponse(400, 'fail', ['record_type' => ['Unsupported record type: '.$recordType]], null);
        }

        // بناء بيانات الاستجابة
        $responseData = [
            'invoice_service_id' => $invoiceService->id, // معرف invoice service
            'pnr_number' => $pnr, // رقم PNR
            'action' => $action, // الإجراء الذي تم تنفيذه (created/updated/cancelled)
            'record_type' => $recordType, // نوع السجل (7A, MA, RF, NT, AMD)
            'service_id' => $parsedData['service_id'], // معرف الخدمة
            'segments_count' => count($parsedData['segments'] ?? []), // عدد Segments
            'passengers_count' => count($parsedData['pax']['packs_details'] ?? []), // عدد المسافرين
            'file_path' => $savedFilePath, // مسار الملف المحفوظ
        ];

        // إرجاع استجابة نجاح مع البيانات
        return $this->apiResponse(200, 'success', null, $responseData);
    }

    /**
     * Search flights without attached profile_id with flexible filters.
     */
    public function searchFlightsWithoutProfile(Request $request)
    {
        $results = $this->invoiceServicesService->searchFlightsWithoutProfile($request->query());

        if ($results === false) {
            return $this->apiResponse(400, 'fail', $this->invoiceServicesService->errors(), null);
        }

        return $this->apiResponse(200, 'success', null, new InvoiceServicesCollection($results, 'flights'));
    }

    /**
     * Attach a profile to multiple invoice services that currently have no profile assigned.
     */
    public function assignProfileToServices(Request $request)
    {
        $result = $this->invoiceServicesService->assignProfileToServices($request->all());

        if ($result === false) {
            return $this->apiResponse(400, 'fail', $this->invoiceServicesService->errors(), null);
        }

        return $this->apiResponse(200, 'success', null, $result);
    }
}
