<?php

namespace App\sys\Repository\Accounting;

use App\Models\Accounting\CostCenter;

class CostCenterRepository
{
    public function create($data)
    {
        $costCenter = new CostCenter;
        $costCenter->name = $data['name'] ?? null;
        $costCenter->cost_center_id = $data['cost_center_id'] ?? null;
        $costCenter->number = $data['number'] ?? null;
        $costCenter->full_number = $data['full_number'] ?? null;
        $costCenter->save();
        $costCenter->load('parent');

        return $costCenter;
    }

    public function update(CostCenter $costCenter, array $data)
    {
        $costCenter->name = $data['name'] ?? $costCenter->name;
        $costCenter->cost_center_id = $data['cost_center_id'] ?? $costCenter->cost_center_id;

        // فقط لو موجودين في data (يعني تم تغيير cost_center_id)
        if (isset($data['number'])) {
            $costCenter->number = $data['number'];
        }
        if (isset($data['full_number'])) {
            $costCenter->full_number = $data['full_number'];
        }

        $costCenter->save();
        $costCenter->load('parent');

        return $costCenter;
    }

    public function findById($id)
    {
        return CostCenter::with('parent')->find($id);
    }

    public function del(array $ids)
    {
        return CostCenter::whereIn('id', $ids)->delete();
    }

    private $columns = [
        'id' => 'id',
        'name' => 'name',
        'cost_center_id' => 'cost_center_id',
        'number' => 'number',
        'full_number' => 'full_number',
    ];

    public function getAll()
    {
        return CostCenter::with('getAllChildren')->whereNull('cost_center_id')->get();
    }

    public function getAllParent()
    {
        return costCenter::with('parent')->whereNull('cost_center_id')->get();
    }

    public function getbyparent($parent)
    {
        return costCenter::where('cost_center_id', $parent)->get();
    }

    public function getPaginated()
    {
        $column = request('sort_by', null);
        $order = request('sort_order', 'asc');
        $name = request('name', null);
        $limit = request('limit', 15);

        return CostCenter::with('parent')
            ->when($name, function ($q, $name) {
                $q->where('name', 'LIKE', "%$name%");
            })->when($column, function ($q, $column) use ($order) {
                if ($column && array_key_exists($column, $this->columns)) {
                    return $q->orderBy($this->columns[$column], $order);
                }
            })->paginate($limit);
    }

    /**
     * Generate next serial number based on parent
     *
     * @param  int|null  $parentId
     * @return array ['number' => int, 'full_number' => string]
     */
    public function generateNextSerial($parentId = null)
    {
        // لو مفيش parent (مستوى أولي)
        if (! $parentId || $parentId == 0) {
            // هات آخر رقم في المستوى الأولي
            $last = CostCenter::where(function ($query) {
                $query->whereNull('cost_center_id');
            })
                ->orderBy('number', 'desc')
                ->first();

            $nextNumber = $last ? ($last->number + 1) : 1;

            // تحقق من عدم تجاوز 9 في المستوى الأولي
            if ($nextNumber > 9) {
                return [
                    'number' => null,
                    'full_number' => null,
                    'error' => 'تم الوصول إلى الحد الأقصى للمستوى الأولي (9)',
                ];
            }

            return [
                'number' => $nextNumber,
                'full_number' => (string) $nextNumber,
            ];
        }

        // لو فيه parent
        $parent = CostCenter::find($parentId);
        if (! $parent) {
            return [
                'number' => null,
                'full_number' => null,
                'error' => 'Parent cost center not found',
            ];
        }

        // هات آخر رقم تحت نفس الأب
        $last = CostCenter::where('cost_center_id', $parentId)
            ->orderBy('number', 'desc')
            ->first();

        $nextNumber = $last ? ($last->number + 1) : 1;

        // بناء full_number من parent.full_number + "-" + number الحالي
        $fullNumber = $parent->full_number.'-'.$nextNumber;

        return [
            'number' => $nextNumber,
            'full_number' => $fullNumber,
        ];
    }
}
