<?php

namespace App\sys\Repository\General;

use App\Models\General\Area;
use App\Models\General\City;

class AreaRepository
{
    private $columns = [
        'id' => 'id',
        'name' => 'name',
        'active' => 'is_active',
        'city_id' => 'city_id',
        'city_name' => 'city_id',
        'created_at' => 'created',
        'updated_at' => 'modified',
    ];

    public function getPaginated()
    {
        $column = request('sort_by', null);
        $order = request('sort_order', 'asc');
        $name = request('name', null);
        $limit = request('limit', 15);

        return Area::when($name, function ($q, $name) {
            $q->whereHas('currentTranslation', function ($q) use ($name) {
                $q->where('name', 'LIKE', "%$name%");
            });
        })->when($column && array_key_exists($column, $this->columns), function ($query) use ($column, $order) {
            if ($column == 'name') {
                $query->leftJoin('area_translation', function ($join) {
                    $join->on('area.id', '=', 'area_translation.area_id')
                        ->where('area_translation.lang_id', app('lang_id'));
                })->orderBy('area_translation.name', $order)->select('area.*');
            } else {
                $query->orderBy($this->columns[$column], $order);
            }
        })->with('currentTranslation', 'city')->paginate($limit);
    }

    public function findByIdOrFail(int $id)
    {
        return Area::with('currentTranslation', 'city')->find($id);
    }

    public function create(array $data)
    {
        $area = new Area;
        $area->city_id = $data['city_id'];
        $area->name = $data['name'];
        $area->is_active = $data['active'];
        $area->save();

        return $area;
    }

    public function update(Area $area, array $data)
    {
        $area->city_id = $data['city_id'] ?? $area->city_id;
        $area->name = $data['name'] ?? $area->name;
        $area->is_active = $data['active'] ?? $area->is_active;
        $area->save();

        return $area;
    }

    public function delete(Area $area)
    {
        return $area->delete();
    }

    public function del(array $ids)
    {
        return Area::whereIn('id', $ids)->delete();
    }

    public function getActive()
    {
        return Area::with('currentTranslation', 'city')->where('is_active', 1)->get();
    }

    public function getByIdWithTranslation($id)
    {
        return Area::with('translations', 'city')->find($id);
    }

    public function getAreasByCity($cityId)
    {
        return Area::with('currentTranslation', 'city')
            ->where('city_id', $cityId)
            ->where('is_active', 1)
            ->get();
    }

    public function getAllChanges($area)
    {
        return $area->audits()
            ->with(['user' => function ($query) {
                $query->select('id', 'name');
            }])
            ->get()
            ->map(function ($audit) {
                $oldValues = $audit->old_values ?? [];
                $newValues = $audit->new_values ?? [];

                $cityIds = array_filter([
                    $oldValues['city_id'] ?? null,
                    $newValues['city_id'] ?? null,
                ]);

                $cities = $cityIds ? City::whereIn('id', $cityIds)
                    ->pluck('name', 'id')
                    ->toArray() : [];

                if (isset($oldValues['city_id']) && isset($cities[$oldValues['city_id']])) {
                    $oldValues['city_name'] = $cities[$oldValues['city_id']];
                }
                if (isset($newValues['city_id']) && isset($cities[$newValues['city_id']])) {
                    $newValues['city_name'] = $cities[$newValues['city_id']];
                }

                return [
                    'audit_id' => $audit->id,
                    'user_id' => $audit->user_id ?? null,
                    'user' => $audit->user ? $audit->user->toArray() : null,
                    'old_values' => $oldValues,
                    'new_values' => $newValues,
                    'changed_at' => $audit->created_at,
                    'event' => $audit->event,
                    'ip_address' => $audit->ip_address,
                    'user_agent' => $audit->user_agent,
                ];
            })
            ->values();
    }
}
