<?php

namespace App\sys\Repository\Profile;

use App\Models\General\Companies;
use App\Models\General\Country;
use App\Models\General\Nationality;
use App\Models\Profile\Customer;

class CustomerRepository
{
    private $columns = [
        'id' => 'id',
        'full_name' => 'full_name',
        'email' => 'email',
        'phone' => 'phone',
        'is_active' => 'is_active',
        'type' => 'type',
        'created_at' => 'created_at',
        'updated_at' => 'updated_at',
    ];

    public function getPaginated($filters = [])
    {
        $query = Customer::with(['nationality.currentTranslation', 'country.currentTranslation', 'company', 'user'])
            ->when(! empty($filters['full_name']), function ($q) use ($filters) {
                $q->where('full_name', 'LIKE', "%{$filters['full_name']}%");
            })
            ->when(! empty($filters['email']), function ($q) use ($filters) {
                $q->where('email', 'LIKE', "%{$filters['email']}%");
            })
            ->when(! empty($filters['phone']), function ($q) use ($filters) {
                $q->where('phone', 'LIKE', "%{$filters['phone']}%");
            })
            ->when(array_key_exists('is_active', $filters), function ($q) use ($filters) {
                $q->where('is_active', $filters['is_active']);
            })
            ->when(array_key_exists('type', $filters) && $filters['type'], function ($q) use ($filters) {
                $q->where('type', $filters['type']);
            })
            ->when(! empty($filters['nationality_id']), function ($q) use ($filters) {
                $q->where('nationality_id', $filters['nationality_id']);
            })
            ->when(! empty($filters['country_id']), function ($q) use ($filters) {
                $q->where('country_id', $filters['country_id']);
            })
            ->when(! empty($filters['company_id']), function ($q) use ($filters) {
                $q->where('company_id', $filters['company_id']);
            });

        $column = $filters['sort_by'] ?? null;
        $order = $filters['sort_order'] ?? 'asc';
        $limit = $filters['limit'] ?? 15;

        $query->when($column && array_key_exists($column, $this->columns), function ($q) use ($column, $order) {
            $q->orderBy($this->columns[$column], $order);
        });

        return $query->paginate($limit);
    }

    public function findByIdOrFail($id)
    {
        return Customer::with(['nationality.currentTranslation', 'country.currentTranslation', 'company', 'user'])->find($id);
    }

    public function create(array $data)
    {
        $item = new Customer;
        $item->full_name = $data['full_name'] ?? null;
        $item->email = $data['email'] ?? null;
        $item->phone = $data['phone'] ?? null;
        $item->nationality_id = $data['nationality_id'] ?? null;
        $item->country_id = $data['country_id'] ?? null;
        $item->passport = $data['passport'] ?? null;
        $item->company_id = $data['company_id'] ?? null;
        $item->user_id = $data['user_id'] ?? null;
        $item->character_status = $data['character_status'] ?? null;
        $item->notes = $data['notes'] ?? null;
        $item->type = $data['type'] ?? null;
        $item->is_active = $data['active'] ?? 1;
        $item->save();

        return $item->load(['nationality.currentTranslation', 'country.currentTranslation', 'company', 'user']);
    }

    public function update(Customer $item, array $data)
    {
        $item->full_name = $data['full_name'] ?? $item->full_name;
        $item->email = $data['email'] ?? $item->email;
        $item->phone = $data['phone'] ?? $item->phone;
        $item->nationality_id = $data['nationality_id'] ?? $item->nationality_id;
        $item->country_id = $data['country_id'] ?? $item->country_id;
        $item->passport = $data['passport'] ?? $item->passport;
        $item->company_id = $data['company_id'] ?? $item->company_id;
        $item->user_id = $data['user_id'] ?? $item->user_id;
        $item->character_status = $data['character_status'] ?? $item->character_status;
        $item->notes = $data['notes'] ?? $item->notes;
        $item->type = $data['type'] ?? $item->type;
        $item->is_active = $data['active'] ?? $item->is_active;
        $item->save();

        return $item->load(['nationality.currentTranslation', 'country.currentTranslation', 'company', 'user']);
    }

    public function delete(Customer $item)
    {
        return $item->delete();
    }

    public function del(array $ids)
    {
        return Customer::whereIn('id', $ids)->delete();
    }

    public function getActive()
    {
        return Customer::with(['nationality.currentTranslation', 'country.currentTranslation', 'company', 'user'])
            ->where('is_active', 1)
            ->get();
    }

    public function getByIdWithTranslation($id)
    {
        return $this->findByIdOrFail($id);
    }

    public function getAllChanges($customer)
    {
        return $customer->audits()
            ->with(['user' => function ($query) {
                $query->select('id', 'name');
            }])
            ->get()
            ->map(function ($audit) {
                $oldValues = $audit->old_values ?? [];
                $newValues = $audit->new_values ?? [];

                $nationalityIds = array_filter([
                    $oldValues['nationality_id'] ?? null,
                    $newValues['nationality_id'] ?? null,
                ]);
                $countryIds = array_filter([
                    $oldValues['country_id'] ?? null,
                    $newValues['country_id'] ?? null,
                ]);
                $companyIds = array_filter([
                    $oldValues['company_id'] ?? null,
                    $newValues['company_id'] ?? null,
                ]);

                $nationalities = $nationalityIds ? Nationality::whereIn('id', $nationalityIds)
                    ->pluck('name', 'id')
                    ->toArray() : [];
                $countries = $countryIds ? Country::whereIn('id', $countryIds)
                    ->pluck('name', 'id')
                    ->toArray() : [];
                $companies = $companyIds ? Companies::whereIn('id', $companyIds)
                    ->pluck('name_company', 'id')
                    ->toArray() : [];

                if (isset($oldValues['nationality_id']) && isset($nationalities[$oldValues['nationality_id']])) {
                    $oldValues['nationality_name'] = $nationalities[$oldValues['nationality_id']];
                }
                if (isset($newValues['nationality_id']) && isset($nationalities[$newValues['nationality_id']])) {
                    $newValues['nationality_name'] = $nationalities[$newValues['nationality_id']];
                }

                if (isset($oldValues['country_id']) && isset($countries[$oldValues['country_id']])) {
                    $oldValues['country_name'] = $countries[$oldValues['country_id']];
                }
                if (isset($newValues['country_id']) && isset($countries[$newValues['country_id']])) {
                    $newValues['country_name'] = $countries[$newValues['country_id']];
                }

                if (isset($oldValues['company_id']) && isset($companies[$oldValues['company_id']])) {
                    $oldValues['company_name'] = $companies[$oldValues['company_id']];
                }
                if (isset($newValues['company_id']) && isset($companies[$newValues['company_id']])) {
                    $newValues['company_name'] = $companies[$newValues['company_id']];
                }

                return [
                    'audit_id' => $audit->id,
                    'user_id' => $audit->user_id ?? null,
                    'user' => $audit->user ? $audit->user->toArray() : null,
                    'old_values' => $oldValues,
                    'new_values' => $newValues,
                    'changed_at' => $audit->created_at,
                    'event' => $audit->event,
                    'ip_address' => $audit->ip_address,
                    'user_agent' => $audit->user_agent,
                ];
            })
            ->values();
    }
}
