<?php

namespace App\sys\Repository\Profile;

use App\Models\General\Companies;
use App\Models\Profile\Customer;
use App\Models\Profile\Profile;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Support\Str;

class ProfileRepository
{
    private $columns = [
        'id' => 'id',
        'profile_number' => 'profile_number',
        'company_id' => 'company_id',
        'user_id' => 'user_id',
        'customer_id' => 'customer_id',
        'arrival_date' => 'arrival_date',
        'departure_date' => 'departure_date',
        'status' => 'status',
        'is_active' => 'is_active',
        'created_at' => 'created_at',
        'updated_at' => 'updated_at',
    ];

    public function getPaginated()
    {
        // Define all filter variables first
        $profileNumber = request('profile_number', null);
        $customerName = request('customer_name', null);
        $nationalityId = request('nationality_id', null);
        $countryId = request('country_id', null);
        $companyId = request('company_id', null);
        $arrivalDateFrom = request('arrival_date_from', null);
        $departureDateFrom = request('departure_date_from', null);
        $createdAt = request('created_at', null);
        $arrivalAirline = request('arrival_airline', null);
        $departureAirline = request('departure_airline', null);
        $status = request('status', null);
        $isActive = request('is_active', null);
        $column = request('sort_by', 'id');
        $order = request('sort_order', 'desc');
        $limit = request('limit', 15);

        return Profile::with([
            'customer.nationality.currentTranslation',
            'customer.country.currentTranslation',
            'customer.company',
            'customer.user',
            'company',
            'user',
            'airlineArrival.city.currentTranslation',
            'airlineDeparture.city.currentTranslation',
        ])->when($profileNumber, function ($q) use ($profileNumber) {
            return $q->where('profile_number', 'like', '%'.$profileNumber.'%');
        })->when($customerName, function ($q) use ($customerName) {
            return $q->whereHas('customer', function ($subQ) use ($customerName) {
                $subQ->where('full_name', 'like', '%'.$customerName.'%');
            });
        })->when($nationalityId, function ($q) use ($nationalityId) {
            return $q->whereHas('customer', function ($subQ) use ($nationalityId) {
                $subQ->where('nationality_id', $nationalityId);
            });
        })->when($countryId, function ($q) use ($countryId) {
            return $q->whereHas('customer', function ($subQ) use ($countryId) {
                $subQ->where('country_id', $countryId);
            });
        })->when($companyId, function ($q) use ($companyId) {
            return $q->where('company_id', $companyId);
        })->when($arrivalDateFrom && $departureDateFrom, function ($q) use ($arrivalDateFrom, $departureDateFrom) {
            return $q->whereDate('arrival_date', '<=', $arrivalDateFrom)
                ->orWhereDate('departure_date', '>=', $departureDateFrom);
        })->when($createdAt, function ($q) use ($createdAt) {
            return $q->whereDate('created_at', $createdAt);
        })->when($arrivalAirline, function ($q) use ($arrivalAirline) {
            return $q->where('arrival_airline', $arrivalAirline);
        })->when($departureAirline, function ($q) use ($departureAirline) {
            return $q->where('departure_airline', $departureAirline);
        })->when($status, function ($q) use ($status) {
            return $q->where('status', $status);
        })->when($isActive, function ($q) use ($isActive) {
            return $q->where('is_active', $isActive);
        })->when($column && array_key_exists($column, $this->columns), function ($q) use ($column, $order) {
            return $q->orderBy($this->columns[$column], $order);
        }, function ($q) use ($order) {
            return $q->orderBy('id', $order);
        })->paginate($limit);
    }

    public function findByIdOrFail($id)
    {
        return Profile::with([
            'customer.nationality.currentTranslation',
            'customer.country.currentTranslation',
            'customer.company',
            'customer.user',
            'company',
            'user',
            'airlineArrival.city.currentTranslation',
            'airlineDeparture.city.currentTranslation',
        ])->find($id);
    }

    public function getBasicProfile($id)
    {
        return Profile::with([
            'accommodationsReservations.accommodation.currentTranslation',
            'invoicesServices.service.currentTranslation',
            'invoicesServices.city',
            'invoicesServices.cityTo',
        ])->find($id);
    }

    public function getByCode($code)
    {
        return Profile::with([
            'customer.nationality.currentTranslation',
            'customer.country.currentTranslation',
            'customer.company',
            'customer.user',
            'company',
            'travelers.nationality.currentTranslation',
            'travelers.currency.currentTranslation',
            'airlineArrival.city.currentTranslation',
            'airlineDeparture.city.currentTranslation',
        ])->where('link_code', $code)->first();
    }

    public function getByTravaliDate($year, $month)
    {
        $lastSerial = Profile::whereYear('arrival_date', $year)
            ->whereMonth('arrival_date', $month)
            ->max('serial_number');

        return $lastSerial ? $lastSerial + 1 : 1;
    }

    public function checkTravel($date, $id)
    {
        if (Profile::where([['id', $id], ['arrival_date', $date]])->exists()) {
            return true;
        }

        return false;
    }

    public function create(array $data)
    {
        $item = new Profile;
        $item->creator_id = auth('api')->id() ?? null;
        $item->profile_number = $data['profile_number'];
        $item->serial_number = $data['serial_number'];
        $item->company_id = $data['company_id'] ?? null;
        $item->user_id = $data['user_id'] ?? null;
        $item->link_code = $this->createCodeShare();
        $item->customer_id = $data['customer_id'] ?? null;
        $item->children_count = $data['children_count'];
        $item->children_no_fee_count = $data['children_no_fee_count'] ?? 0;
        $item->adults_count = $data['adults_count'];
        $item->arrival_airline = $data['arrival_airline'];
        $item->arrival_flight_number = $data['arrival_flight_number'];
        $item->departure_airline = $data['departure_airline'];
        $item->departure_flight_number = $data['departure_flight_number'];
        $item->arrival_date = $data['arrival_date'];
        $item->arrival_time = $data['arrival_time'] ?? null;
        $item->departure_date = $data['departure_date'];
        $item->departure_time = $data['departure_time'] ?? null;
        $item->trip_duration_days = Carbon::parse($data['arrival_date'])->diffInDays(Carbon::parse($data['departure_date'])) + 1;
        $item->status = 'pending';
        $item->is_active = $data['active'];
        $item->customer_notes = $data['customer_notes'] ?? null;
        $item->employee_notes = $data['employee_notes'] ?? null;

        $item->save();

        return $item->load([
            'customer.nationality.currentTranslation',
            'customer.country.currentTranslation',
            'customer.company',
            'customer.user',
            'company',
            'user',
            'airlineArrival.city.currentTranslation',
            'airlineDeparture.city.currentTranslation',
        ]);
    }

    private function createCodeShare()
    {
        do {
            $code = Str::upper(Str::random(9)).strtoupper(substr(uniqid(), -4));
        } while (Profile::where('link_code', $code)->exists());

        return $code;
    }

    public function update(Profile $item, array $data)
    {
        $item->profile_number = $data['profile_number'] ?? $item->profile_number;
        $item->serial_number = $data['serial_number'] ?? $item->serial_number;
        $item->company_id = $data['company_id'] ?? $item->company_id;
        $item->customer_id = $data['customer_id'] ?? $item->customer_id;
        $item->children_count = $data['children_count'] ?? $item->children_count;
        $item->children_no_fee_count = $data['children_no_fee_count'] ?? $item->children_no_fee_count;
        $item->adults_count = $data['adults_count'] ?? $item->adults_count;
        $item->arrival_airline = $data['arrival_airline'] ?? $item->arrival_airline;
        $item->arrival_flight_number = $data['arrival_flight_number'] ?? $item->arrival_flight_number;
        $item->departure_airline = $data['departure_airline'] ?? $item->departure_airline;
        $item->departure_flight_number = $data['departure_flight_number'] ?? $item->departure_flight_number;
        $item->arrival_date = $data['arrival_date'] ?? $item->arrival_date;
        $item->arrival_time = $data['arrival_time'] ?? $item->arrival_time;
        $item->departure_date = $data['departure_date'] ?? $item->departure_date;
        $item->departure_time = $data['departure_time'] ?? $item->departure_time;
        $item->trip_duration_days = Carbon::parse($data['arrival_date'] ?? $item->departure_date)
            ->diffInDays(Carbon::parse($data['departure_date'] ?? $item->arrival_date)) + 1;
        $item->status = $data['status'] ?? $item->status;
        $item->is_active = $data['active'] ?? $item->is_active;
        $item->customer_notes = $data['customer_notes'] ?? $item->customer_notes;
        $item->employee_notes = $data['employee_notes'] ?? $item->employee_notes;

        $item->save();

        return $item->load([
            'customer.nationality.currentTranslation',
            'customer.country.currentTranslation',
            'customer.company',
            'customer.user',
            'company',
            'user',
            'airlineArrival.city.currentTranslation',
            'airlineDeparture.city.currentTranslation',
        ]);
    }

    public function del(array $ids)
    {
        return Profile::whereIn('id', $ids)->delete();
    }

    public function getActive()
    {
        return Profile::with([
            'customer.nationality.currentTranslation',
            'customer.country.currentTranslation',
            'customer.company',
            'customer.user',
            'company',
            'user',
            'airlineArrival.city.currentTranslation',
            'airlineDeparture.city.currentTranslation',
        ])
            ->where('is_active', 1)
            ->paginate(15);
    }

    public function getByIdWithTranslation($id)
    {
        return $this->findByIdOrFail($id);
    }

    public function getAllChanges($profile)
    {
        return $profile->audits()
            ->with(['user' => function ($query) {
                $query->select('id', 'name');
            }])
            ->get()
            ->map(function ($audit) {
                $oldValues = $audit->old_values ?? [];
                $newValues = $audit->new_values ?? [];

                $companyIds = array_filter([
                    $oldValues['company_id'] ?? null,
                    $newValues['company_id'] ?? null,
                ]);
                $customerIds = array_filter([
                    $oldValues['customer_id'] ?? null,
                    $newValues['customer_id'] ?? null,
                ]);

                $companies = $companyIds ? Companies::whereIn('id', $companyIds)
                    ->pluck('name_company', 'id')
                    ->toArray() : [];
                $customers = $customerIds ? Customer::whereIn('id', $customerIds)
                    ->pluck('full_name', 'id')
                    ->toArray() : [];

                if (isset($oldValues['company_id']) && isset($companies[$oldValues['company_id']])) {
                    $oldValues['company_name'] = $companies[$oldValues['company_id']];
                    unset($oldValues['company']);
                }
                if (isset($newValues['company_id']) && isset($companies[$newValues['company_id']])) {
                    $newValues['company_name'] = $companies[$newValues['company_id']];
                    unset($newValues['company']);
                }

                if (isset($oldValues['customer_id']) && isset($customers[$oldValues['customer_id']])) {
                    $oldValues['customer_name'] = $customers[$oldValues['customer_id']];
                    unset($oldValues['customer']);
                }
                if (isset($newValues['customer_id']) && isset($customers[$newValues['customer_id']])) {
                    $newValues['customer_name'] = $customers[$newValues['customer_id']];
                    unset($newValues['customer']);
                }

                return [
                    'audit_id' => $audit->id,
                    'user_id' => $audit->user_id ?? null,
                    'user' => $audit->user ? $audit->user->toArray() : null,
                    'old_values' => $oldValues,
                    'new_values' => $newValues,
                    'changed_at' => $audit->created_at,
                    'event' => $audit->event,
                    'ip_address' => $audit->ip_address,
                    'user_agent' => $audit->user_agent,
                ];
            })
            ->values();
    }

    public function getColumnChanges($profile, $columnName)
    {
        return $profile->audits()
            ->get()
            ->filter(function ($audit) use ($columnName) {
                $oldValues = $audit->old_values ?? [];
                $newValues = $audit->new_values ?? [];

                return array_key_exists($columnName, $oldValues) ||
                    array_key_exists($columnName, $newValues);
            })
            ->map(function ($audit) use ($columnName) {
                $user = null;
                if ($audit->user_id) {
                    $userModel = User::find($audit->user_id);
                    $user = $userModel ? $userModel->toArray() : null;
                }

                $oldValues = $audit->old_values ?? [];
                $newValues = $audit->new_values ?? [];

                if ($columnName === 'company_id' || $columnName === 'customer_id') {
                    $companyIds = array_filter([
                        $oldValues['company_id'] ?? null,
                        $newValues['company_id'] ?? null,
                    ]);
                    $customerIds = array_filter([
                        $oldValues['customer_id'] ?? null,
                        $newValues['customer_id'] ?? null,
                    ]);

                    $companies = $companyIds ? Companies::whereIn('id', $companyIds)
                        ->pluck('name_company', 'id')
                        ->toArray() : [];
                    $customers = $customerIds ? Customer::whereIn('id', $customerIds)
                        ->pluck('full_name', 'id')
                        ->toArray() : [];

                    if ($columnName === 'company_id') {
                        $oldValue = isset($oldValues['company_id']) ? $companies[$oldValues['company_id']] ?? null : null;
                        $newValue = isset($newValues['company_id']) ? $companies[$newValues['company_id']] ?? null : null;
                    } elseif ($columnName === 'customer_id') {
                        $oldValue = isset($oldValues['customer_id']) ? $customers[$oldValues['customer_id']] ?? null : null;
                        $newValue = isset($newValues['customer_id']) ? $customers[$newValues['customer_id']] ?? null : null;
                    }
                } else {
                    $oldValue = $oldValues[$columnName] ?? null;
                    $newValue = $newValues[$columnName] ?? null;
                }

                return [
                    'audit_id' => $audit->id,
                    'user_id' => $audit->user_id ?? null,
                    'user' => $user['name'] ?? null,
                    'old_value' => $oldValue,
                    'new_value' => $newValue,
                    'changed_at' => $audit->created_at,
                    'event' => $audit->event,
                    'ip_address' => $audit->ip_address,
                    'user_agent' => $audit->user_agent,
                ];
            })
            ->values();
    }

    public function getProfielStatus($profile)
    {
        return Profile::find('id', $profile)?->status;
    }
}
