<?php

namespace App\sys\Repository\Profile;

use App\Models\General\Currency;
use App\Models\General\Nationality;
use App\Models\Profile\ProfileTraveler;
use Illuminate\Support\Facades\DB;
use OwenIt\Auditing\Models\Audit;

class TravelerRepository
{
    private $columns = [
        'id' => 'id',
        'profile_id' => 'profile_id',
        'nationality_id' => 'nationality_id',
        'currency_id' => 'currency_id',
        'type' => 'type',
        'unit_price' => 'unit_price',
        'created_at' => 'created_at',
        'updated_at' => 'updated_at',
    ];

    public function getPaginated($filters = [])
    {
        $query = ProfileTraveler::with(['nationality.currentTranslation', 'currency.currentTranslation', 'profile']);

        if (! empty($filters['profile_id'])) {
            $query->where('profile_id', $filters['profile_id']);
        }
        if (! empty($filters['type'])) {
            $query->where('type', $filters['type']);
        }
        if (! empty($filters['nationality_id'])) {
            $query->where('nationality_id', $filters['nationality_id']);
        }
        if (! empty($filters['currency_id'])) {
            $query->where('currency_id', $filters['currency_id']);
        }

        $column = $filters['sort_by'] ?? null;
        $order = $filters['sort_order'] ?? 'asc';
        $limit = $filters['limit'] ?? 15;
        if ($column && array_key_exists($column, $this->columns)) {
            $query->orderBy($this->columns[$column], $order);
        }

        return $query->paginate($limit);
    }

    public function findByIdOrFail($id)
    {
        return ProfileTraveler::where('profile_id', $id)
            ->with([
                'nationality.currentTranslation',
                'currency.currentTranslation',
                'profile',
                'tax.tax.currentTranslation',
            ])->get();
    }

    public function find($id)
    {
        return ProfileTraveler::with(['nationality.currentTranslation', 'currency.currentTranslation', 'profile'])->find($id);
    }

    public function delete(ProfileTraveler $item)
    {
        return $item->delete();
    }

    public function del(array $ids)
    {
        return ProfileTraveler::whereIn('id', $ids)->delete();
    }

    public function getTravelerNationalities($profileId)
    {
        $travelers = ProfileTraveler::where('profile_id', $profileId)
            ->with('nationality')
            ->get()
            ->groupBy('nationality_id')
            ->map(function ($group) {
                $adultCount = $group->where('type', 'adult')->sum('count');
                $childCount = $group->where('type', 'child')->sum('count');
                $totalCount = $adultCount + $childCount;

                return [
                    'nationality_id' => $group->first()->nationality_id,
                    'nationality_name' => $group->first()->nationality->name ?? null,
                    'adult_count' => $adultCount,
                    'child_count' => $childCount,
                    'total_count' => $totalCount,
                    'traveler_count' => $totalCount,
                ];
            })
            ->values();

        return $travelers;
    }

    public function createMultiple(array $data)
    {
        $travelers = [];

        foreach ($data['travelers'] as $travelerData) {
            $traveler = new ProfileTraveler;
            $traveler->profile_id = $data['profile_id'];
            $traveler->nationality_id = $travelerData['nationality_id'];
            $traveler->currency_id = $travelerData['currency_id'];
            $traveler->type = $travelerData['type'];
            $traveler->unit_price = $travelerData['unit_price'] ?? 0;
            $traveler->count = $travelerData['count'] ?? 0;
            $net = (float) $traveler->unit_price * (int) $traveler->count;
            $traveler->net_total = $net;

            // taxes
            $taxIds = $travelerData['tax_id'] ?? ($data['tax_id'] ?? []);
            $totalTax = 0.0;
            if (is_array($taxIds) && count($taxIds) > 0) {
                $rates = \App\Models\General\TaxRate::whereIn('id', $taxIds)->get(['id', 'percentage']);
                foreach ($rates as $rate) {
                    $totalTax += round(($net * ((float) $rate->percentage)) / 100, 2);
                }
            }
            $traveler->tax = $totalTax;
            $traveler->total_price = $net + $totalTax;
            $traveler->save();

            // persist tax breakdown
            if (is_array($taxIds) && count($taxIds) > 0) {
                foreach ($rates as $rate) {
                    \App\Models\Profile\ProfileTravelerTax::create([
                        'profile_id' => $traveler->profile_id,
                        'profile_travelers_id' => $traveler->id,
                        'tax_rate_id' => $rate->id,
                        'amount' => round(($net * ((float) $rate->percentage)) / 100, 2),
                    ]);
                }
            }
            $travelers[] = $traveler;
        }

        return ProfileTraveler::with(['nationality.currentTranslation', 'currency.currentTranslation', 'profile', 'tax.tax.currentTranslation'])
            ->whereIn('id', collect($travelers)->pluck('id'))
            ->get();
    }

    public function updateMultiple(array $data)
    {
        return DB::transaction(function () use ($data) {
            // Get old records for audit
            $oldTravelers = ProfileTraveler::where('profile_id', $data['profile_id'])->get();

            ProfileTraveler::where('profile_id', $data['profile_id'])->delete();

            $newTravelers = $this->createMultiple($data);

            // Create a custom batch audit entry
            Audit::create([
                'auditable_type' => 'ProfileTraveler',
                'auditable_id' => $data['profile_id'], // Use profile_id as reference
                'event' => 'updated',
                'old_values' => ['travelers' => $oldTravelers->toArray()],
                'new_values' => ['travelers' => $newTravelers->toArray()],
                // TODO: add the user_id after we do the auth
                'user_id' => null,
                'user_type' => null,
                'url' => request()->fullUrl(),
                'ip_address' => request()->ip(),
                'user_agent' => request()->userAgent(),
                'tags' => 'null',
                'created_at' => now(),
                'updated_at' => now(),
            ]);

            return $newTravelers;
        });
    }

    public function getAllChanges($profileId)
    {
        $audits = Audit::where('auditable_type', 'ProfileTraveler')
            ->where('auditable_id', $profileId)
            ->with(['user' => function ($query) {
                $query->select('id', 'name');
            }])
            ->orderBy('created_at', 'desc')
            ->get();

        return $audits->map(function ($audit) {
            $oldValues = $audit->old_values ?? [];
            $newValues = $audit->new_values ?? [];

            $nationalityIds = [];
            $currencyIds = [];

            if (isset($oldValues['travelers'])) {
                foreach ($oldValues['travelers'] as $traveler) {
                    if (isset($traveler['nationality_id'])) {
                        $nationalityIds[] = $traveler['nationality_id'];
                    }
                    if (isset($traveler['currency_id'])) {
                        $currencyIds[] = $traveler['currency_id'];
                    }
                }
            }

            if (isset($newValues['travelers'])) {
                foreach ($newValues['travelers'] as $traveler) {
                    if (isset($traveler['nationality_id'])) {
                        $nationalityIds[] = $traveler['nationality_id'];
                    }
                    if (isset($traveler['currency_id'])) {
                        $currencyIds[] = $traveler['currency_id'];
                    }
                }
            }

            $nationalityIds = array_filter(array_unique($nationalityIds));
            $currencyIds = array_filter(array_unique($currencyIds));

            $nationalities = $nationalityIds ? Nationality::whereIn('id', $nationalityIds)
                ->pluck('name', 'id')
                ->toArray() : [];
            $currencies = $currencyIds ? Currency::whereIn('id', $currencyIds)
                ->pluck('name', 'id')
                ->toArray() : [];

            $processedOldValues = $oldValues;
            if (isset($oldValues['travelers'])) {
                foreach ($processedOldValues['travelers'] as &$traveler) {
                    if (isset($traveler['nationality_id']) && isset($nationalities[$traveler['nationality_id']])) {
                        $traveler['nationality_name'] = $nationalities[$traveler['nationality_id']];
                    }
                    if (isset($traveler['currency_id']) && isset($currencies[$traveler['currency_id']])) {
                        $traveler['currency_name'] = $currencies[$traveler['currency_id']];
                    }
                }
            }

            $processedNewValues = $newValues;
            if (isset($newValues['travelers'])) {
                foreach ($processedNewValues['travelers'] as &$traveler) {
                    if (isset($traveler['nationality_id']) && isset($nationalities[$traveler['nationality_id']])) {
                        $traveler['nationality_name'] = $nationalities[$traveler['nationality_id']];
                    }
                    if (isset($traveler['currency_id']) && isset($currencies[$traveler['currency_id']])) {
                        $traveler['currency_name'] = $currencies[$traveler['currency_id']];
                    }
                }
            }

            return [
                'audit_id' => $audit->id,
                'user_id' => $audit->user_id ?? null,
                'user' => $audit->user ? $audit->user->name : null,
                'old_values' => $processedOldValues ?: null,
                'new_values' => $processedNewValues ?: null,
                'changed_at' => $audit->created_at,
                'event' => $audit->event,
                'ip_address' => $audit->ip_address,
                'user_agent' => $audit->user_agent,
            ];
        });
    }

    public function getAmountByCurrnecy($profileId, $currencyId)
    {
        return ProfileTraveler::where([
            ['profile_id', $profileId],
            ['currency_id', $currencyId],
        ])->sum('total_price');
    }

    public function getByprofile($profileId)
    {
        return ProfileTraveler::select('currency_id', DB::raw('SUM(total_price) as total'))
            ->where('profile_id', $profileId)
            ->groupBy('currency_id')
            ->get();
    }

    public function selectedCurencyToProfile($profileId)
    {
        return ProfileTraveler::select(
            'currencies.id as id',
            'currencies.name as name',
        )
            ->join('currencies', 'currencies.id', '=', 'pr_profile_travelers.currency_id')
            ->where('pr_profile_travelers.profile_id', $profileId)
            ->distinct()
            ->get();
    }
}
