<?php

namespace App\sys\Services\Accounting;

use App\sys\Repository\Accounting\InvoiceRepository;
use App\sys\Services;
use Illuminate\Support\Facades\Validator;

class InvoiceService extends Services
{
    protected InvoiceRepository $invoiceRepository;

    public function __construct(InvoiceRepository $invoiceRepository)
    {
        $this->invoiceRepository = $invoiceRepository;
    }

    public function getPaginated()
    {
        return $this->invoiceRepository->getPaginated();
    }

    public function getById(int $id)
    {
        $validator = Validator::make(['id' => $id], [
            'id' => ['required', 'integer', 'exists:in_invoice,id'],
        ]);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->invoiceRepository->findById($id);
    }

    public function getByProfileId(int $profileId)
    {
        $validator = Validator::make(['profile_id' => $profileId], [
            'profile_id' => ['required', 'integer', 'exists:pr_profile,id'],
        ]);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->invoiceRepository->getByProfileId($profileId);
    }

    public function getByNextDate(string $nextDate)
    {
        $validator = Validator::make(['next_date' => $nextDate], [
            'next_date' => ['required', 'date'],
        ]);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->invoiceRepository->getByNextDate($nextDate);
    }

    public function create(array $request)
    {
        $rules = [
            'profile_id' => ['sometimes', 'nullable', 'integer', 'exists:pr_profile,id'],
            'customer_id' => ['required', 'integer', 'exists:pr_customer,id'],
            'company_id' => ['required', 'integer', 'exists:companies,id'],
            'invoice_date' => ['required', 'date'],
            'invoice_price' => ['required', 'numeric', 'min:0'],
            'paid_price' => ['sometimes', 'numeric', 'min:0'],
            'remaining_price' => ['sometimes', 'numeric'],
            'invoice_type' => ['sometimes', 'integer'],
            'serial_num' => ['nullable', 'string', 'max:50'],
            'recip_date' => ['nullable', 'date'],
            'pay_duration' => ['nullable', 'integer'],
            'invoice_date_due' => ['nullable', 'date'],
            'paid_date' => ['nullable', 'date'],
            'worthy_days' => ['nullable', 'integer'],
            'notes' => ['nullable', 'string'],
            'conditions' => ['nullable', 'string'],
            'next_paid_date' => ['nullable', 'date'],
        ];

        $validator = Validator::make($request, $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->invoiceRepository->create($request);
    }

    public function update(array $request)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:in_invoice,id'],
            'profile_id' => ['sometimes', 'nullable', 'integer', 'exists:pr_profile,id'],
            'customer_id' => ['sometimes', 'integer', 'exists:pr_customer,id'],
            'company_id' => ['sometimes', 'integer', 'exists:companies,id'],
            'invoice_date' => ['sometimes', 'date'],
            'invoice_price' => ['sometimes', 'numeric', 'min:0'],
            'paid_price' => ['sometimes', 'numeric', 'min:0'],
            'remaining_price' => ['sometimes', 'numeric'],
            'invoice_type' => ['sometimes', 'integer'],
            'recip_date' => ['nullable', 'date'],
            'pay_duration' => ['nullable', 'integer'],
            'invoice_date_due' => ['nullable', 'date'],
            'paid_date' => ['nullable', 'date'],
            'worthy_days' => ['nullable', 'integer'],
            'notes' => ['nullable', 'string'],
            'conditions' => ['nullable', 'string'],
            'next_paid_date' => ['nullable', 'date'],
        ];

        $validator = Validator::make($request, $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        $invoice = $this->getById($request['id']);
        if (! $invoice) {
            return false;
        }

        return $this->invoiceRepository->update($invoice, $request);
    }

    public function del(array $ids)
    {
        $validator = Validator::make(['ids' => $ids], [
            'ids' => ['required', 'array', 'min:1'],
            'ids.*' => ['required', 'integer', 'exists:in_invoice,id'],
        ]);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->invoiceRepository->del($ids);
    }
}
