<?php

namespace App\sys\Services\Accounting;

use App\Models\Accounting\Paytype;
use App\sys\Enums\PaymentTypes;
use App\sys\Repository\Accounting\PaytypeRepository;
use App\sys\Services;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;

class PaytypeService extends Services
{
    protected PaytypeRepository $paytypeRepository;

    public function __construct(PaytypeRepository $paytypeRepository)
    {
        $this->paytypeRepository = $paytypeRepository;
    }

    public function getPaginatedPaytypes()
    {
        return $this->paytypeRepository->getPaginated();
    }

    public function getByCurrnecy($currnecy)
    {
        return $this->paytypeRepository->getByCurrnecy($currnecy);
    }

    public function getPaytypeById(int $id)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:pay_type,id'],
        ];
        $validator = Validator::make(['id' => $id], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->paytypeRepository->findByIdOrFail($id);
    }

    public function createPaytype($request)
    {
        if (! $this->validatePaytype($request)) {
            return false;
        }

        $paytype = $this->paytypeRepository->create($request);
        $this->syncTranslations($paytype);

        return $paytype;
    }

    public function updatePaytype($request)
    {
        if (! $this->validUpdated($request)) {
            return false;
        }

        $result = $this->paytypeRepository->update($request);

        if (! $result) {
            $this->setError(['id' => ['Paytype not found']]);

            return false;
        }

        return $result;
    }

    private function validatePaytype($request)
    {
        $rules = [
            'title' => ['required', 'string', 'max:255', 'unique:pay_type,title'],
            'type' => ['required', 'string', Rule::in($this->getPaymentTypeValues())],
            'accounting' => ['nullable', 'array'],
        ];
        $validator = Validator::make($request, $rules);
        $validator->after(function ($validator) use ($request) {
            $type = $request['type'] ?? null;
            $accounts = $request['accounting'] ?? null;

            $this->validateAccountingData($validator, $accounts, $type, null, true);
        });
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return true;
    }

    private function validUpdated($request)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:pay_type,id'],
            'title' => ['sometimes', 'string', 'max:255', 'unique:pay_type,title,'.$request['id']],
            'type' => ['sometimes', 'string', Rule::in($this->getPaymentTypeValues())],
            'accounting' => ['sometimes', 'nullable', 'array'],
        ];
        $validator = Validator::make($request, $rules);
        $validator->after(function ($validator) use ($request) {
            $type = $request['type'] ?? null;
            $accounts = $request['accounting'] ?? null;
            $paytypeId = $request['id'] ?? null;
            // في التعديل برضه نطبّق نفس منطق الإلزام لو النوع يتطلب بيانات محاسبية
            $this->validateAccountingData($validator, $accounts, $type, $paytypeId, true);
        });
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return true;
    }

    public function del(array $ids)
    {
        $result = $this->paytypeRepository->del($ids);

        if (is_array($result) && ! $result['success']) {
            $this->setError(['delete' => [$result['message']]]);

            return false;
        }

        return $result;
    }

    public function getByidWithTranslation($id)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:pay_type,id'],
        ];

        $validator = Validator::make(['id' => $id], $rules);

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }
        $paytype = $this->paytypeRepository->getByIdWithTranslation($id);

        if (! $paytype) {
            $this->setError(['id' => ['Paytype not found']]);

            return false;
        }

        return $this->syncTranslations($paytype, true);
    }

    public function getAllChanges($paytype)
    {
        return $this->paytypeRepository->getAllChanges($paytype);
    }

    public function getTOSupplier()
    {
        return $this->paytypeRepository->getTOSupplier();
    }

    private function getPaymentTypeValues(): array
    {
        return array_map(fn (PaymentTypes $type) => $type->value, PaymentTypes::cases());
    }

    private function validateAccountingData($validator, $accounts, ?string $type = null, ?int $paytypeId = null, bool $enforceRequirement = false): void
    {
        $effectiveType = $type;

        if (! $effectiveType && $paytypeId) {
            $effectiveType = Paytype::where('id', $paytypeId)->value('type');
        }

        // لازم يبعت الـ accounting لو النوع بنك أو نسب طرق الدفع
        $requiresAccounting = in_array($effectiveType, [
            PaymentTypes::BANK->value,
            PaymentTypes::PAYMENT_METHOD_RATES->value,
        ], true);

        if ($enforceRequirement && $requiresAccounting && (empty($accounts) || ! is_array($accounts))) {
            $validator->errors()->add('accounting', 'يجب إرسال بيانات المحاسبة عند اختيار نوع نسب طرق الدفع');

            return;
        }

        if (empty($accounts) || ! is_array($accounts)) {
            return;
        }

        $sentKeys = [];
        $missingFieldsErrors = [];

        foreach ($accounts as $index => $item) {
            // في حالة البنك: لازم currency_id + tree_accounting_transfer_id فقط
            // في باقي الأنواع (ومنها نسب طرق الدفع) نحتاج الحقول القديمة كلها
            if ($effectiveType === PaymentTypes::BANK->value) {
                $required = ['currency_id', 'tree_accounting_transfer_id'];
            } else {
                $required = ['currency_id', 'percentage', 'tree_accounting_transfer_id', 'tree_account_commission_id'];
            }
            $missing = array_diff($required, array_keys($item));

            if ($missing) {
                $missingFieldsErrors[] = "العنصر رقم {$index} ناقص: ".implode(', ', $missing);

                continue;
            }

            $sentKeys[] = $item['currency_id'];
        }

        if ($missingFieldsErrors) {
            foreach ($missingFieldsErrors as $err) {
                $validator->errors()->add('accounting', $err);
            }

            return;
        }

        // منع تكرار نفس العملة لكل الأنواع ما عدا نسب طرق الدفع
        if ($effectiveType !== PaymentTypes::PAYMENT_METHOD_RATES->value
            && count($sentKeys) !== count(array_unique($sentKeys))) {
            $validator->errors()->add('accounting', 'لا يمكن تكرار نفس العملة');
        }
    }
}
