<?php

namespace App\sys\Services\General;

use App\sys\Repository\General\GuideLanguageRepository;
use App\sys\Services;
use Illuminate\Support\Facades\Validator;

class GuideLanguageService extends Services
{
    protected GuideLanguageRepository $repository;

    public function __construct(GuideLanguageRepository $repository)
    {
        $this->repository = $repository;
    }

    public function getPaginated()
    {
        return $this->repository->getPaginated();
    }

    public function getById(int $id)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:guide_language,id'],
        ];
        $validator = Validator::make(['id' => $id], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->repository->findByIdOrFail($id);
    }

    public function create($request)
    {
        $rules = [
            'name' => ['required', 'string', 'max:255', 'unique:guide_language,name'],
            'active' => ['required', 'in:0,1'],
        ];
        $validator = Validator::make($request, $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }
        $model = $this->repository->create($request);
        $this->syncTranslations($model);

        return $model;
    }

    public function update($request)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:guide_language,id'],
            'name' => ['sometimes', 'string', 'max:255', 'unique:guide_language,name,'.($request['id'] ?? 'null')],
            'active' => ['sometimes', 'in:0,1'],
        ];
        $validator = Validator::make($request, $rules);
        $model = $this->repository->findByIdOrFail($request['id']);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->repository->update($model, $request);
    }

    public function del(array $ids)
    {
        return $this->repository->del($ids);
    }

    public function getActive()
    {
        return $this->repository->getActive();
    }

    public function getByIdWithTranslation($id)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:guide_language,id'],
        ];
        $validator = Validator::make(['id' => $id], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }
        $model = $this->repository->getByIdWithTranslation($id);

        return $this->syncTranslations($model, true);
    }
}
