<?php

namespace App\sys\Services\General;

use App\sys\Repository\General\GuideLanguageTranslationRepository;
use App\sys\Services;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;

class GuideLanguageTranslationService extends Services
{
    private $translationRepository;

    public function __construct()
    {
        $this->translationRepository = new GuideLanguageTranslationRepository;
    }

    public function save(array $data, $id)
    {
        $rules = ['guide_language_id' => 'required|integer|exists:guide_language,id'];
        $combinations = [];
        $customErrors = [];

        foreach ($data as $index => $row) {
            $rowId = $row['id'] ?? null;
            $languageId = $row['language_id'] ?? null;
            $rules["$index.id"] = 'sometimes|nullable|exists:guide_language_translation,id';
            $rules["$index.language_id"] = [
                Rule::requiredIf(empty($rowId)),
                'exists:languages,id',
                Rule::unique('guide_language_translation', 'lang_id')
                    ->where(fn ($query) => $query->where('guide_language_id', $id))
                    ->ignore($rowId),
            ];
            $rules["$index.name"] = 'nullable|string|max:255';

            if ($languageId) {
                $key = $id.'-'.$languageId;
                if (in_array($key, $combinations)) {
                    $customErrors["$index.language_id"] = 'Same language distorted with the same Guide Language.';
                } else {
                    $combinations[] = $key;
                }
            }
        }

        $validator = Validator::make(array_merge($data, ['guide_language_id' => $id]), $rules);

        $validator->after(function ($validator) use ($customErrors) {
            foreach ($customErrors as $field => $message) {
                $validator->errors()->add($field, $message);
            }
        });

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->translationRepository->save($data, $id);
    }

    public function get($id)
    {
        return $this->translationRepository->get($id);
    }

    public function del($id)
    {
        $rules = [
            'id' => 'required|integer|exists:guide_language_translation,id',
        ];
        $validator = \Validator::make(['id' => $id], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->translationRepository->del($id);
    }
}
