<?php

namespace App\sys\Services\General;

use App\sys\Enums\TaxRateAccountMapping;
use App\sys\Helper;
use App\sys\Repository\General\TaxRateRepository;
use App\sys\Services;
use Illuminate\Support\Facades\Validator;

class TaxRateService extends Services
{
    protected TaxRateRepository $taxRateRepository;

    public function __construct(TaxRateRepository $taxRateRepository)
    {
        $this->taxRateRepository = $taxRateRepository;
    }

    public function getPaginatedTaxRates()
    {
        return $this->taxRateRepository->getPaginated();
    }

    public function getTaxRateById(int $id)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:taxs_rate,id'],
        ];
        $validator = Validator::make(['id' => $id], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->taxRateRepository->findByIdOrFail($id);
    }

    public function createTaxRate($request)
    {
        $rules = [
            'name' => ['required', 'string', 'max:255'],
            'percentage' => ['required', 'numeric', 'max:255'],
            'type' => ['required', 'string', 'in:general,revenue'],
            'active' => ['required', 'in:0,1'],
            'accounting' => ['sometimes', 'array'],
            'accounting.*.input_name' => ['required_with:accounting'],
            'accounting.*.currency_id' => ['required_with:accounting', 'exists:currencies,id'],
            'accounting.*.tree_account_id' => ['required_with:accounting', 'exists:tree_accounting,id'],
        ];

        $validator = Validator::make($request, $rules);

        $validator->after(
            function ($validator) use ($request) {
                if (! empty($request['accounting']) && is_array($request['accounting'])) {
                    \App\sys\Helper::validateCurrencyInputCombinations($validator, $request, TaxRateAccountMapping::class, 'accounting', [], 'ضرائب');
                }
            }
        );

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }
        $domain = $this->taxRateRepository->create($request);
        $this->syncTranslations($domain);

        return $domain;
    }

    public function updateTaxRate($request)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:taxs_rate,id'],
            'name' => ['sometimes', 'string', 'max:255'],
            'percentage' => ['sometimes', 'numeric', 'max:255'],
            'type' => ['sometimes', 'string', 'in:general,revenue'],
            'active' => ['sometimes', 'in:0,1'],
            'accounting' => ['sometimes', 'array'],
            'accounting.*.input_name' => ['required_with:accounting'],
            'accounting.*.currency_id' => ['required_with:accounting', 'exists:currencies,id'],
            'accounting.*.tree_account_id' => ['sometimes', 'exists:tree_accounting,id'],
        ];
        $validator = Validator::make($request, $rules);

        $validator->after(
            function ($validator) use ($request) {
                if (! empty($request['accounting']) && is_array($request['accounting'])) {
                    \App\sys\Helper::validateCurrencyInputCombinations($validator, $request, TaxRateAccountMapping::class, 'accounting', [], 'ضرائب');
                }
            }
        );

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }
        $taxRate = $this->taxRateRepository->findByIdOrFail($request['id']);

        return $this->taxRateRepository->update($taxRate, $request);
    }

    public function deleteTaxRate(int $id)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:taxs_rate,id'],
        ];
        $validator = Validator::make(['id' => $id], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }
        $taxRate = $this->taxRateRepository->findByIdOrFail($id);

        return $this->taxRateRepository->delete($taxRate);
    }

    public function del(array $ids)
    {
        return $this->taxRateRepository->del($ids);
    }

    public function getActive()
    {
        return $this->taxRateRepository->getActive();
    }

    public function getByCurrency(int $currencyId)
    {
        $rules = [
            'currency_id' => ['required', 'integer', 'exists:currencies,id'],
        ];
        $validator = Validator::make(['currency_id' => $currencyId], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->taxRateRepository->getTaxsByCurrency($currencyId);
    }

    public function getByIdWithTranslation($id)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:taxs_rate,id'],
        ];
        $validator = Validator::make(['id' => $id], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }
        $taxRate = $this->taxRateRepository->getByIdWithTranslation($id);

        return $this->syncTranslations($taxRate, true);
    }

    public function getCurrencyWithtree()
    {
        return Helper::getCurrencyWithtree(TaxRateAccountMapping::class);
    }

    public function getAllChanges($taxRate)
    {
        return $this->taxRateRepository->getAllChanges($taxRate);
    }
}
