<?php

namespace App\sys\Services\Profile;

use App\Models\Profile\DailyPrograms;
use App\sys\Repository\Profile\DailyProgramsRepository;
use App\sys\Services;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;

class DailyProgramsService extends Services
{
    protected DailyProgramsRepository $repository;

    public function __construct(DailyProgramsRepository $repository)
    {
        $this->repository = $repository;
    }

    public function getById(int $id)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:pr_daily_programs,id'],
        ];
        $validator = Validator::make(['id' => $id], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->repository->findByIdOrFail($id);
    }

    public function getByProfileId(int $profileId)
    {
        $rules = [
            'profile_id' => ['required', 'integer', 'exists:pr_profile,id'],
        ];
        $validator = Validator::make(['profile_id' => $profileId], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->repository->getByProfileId($profileId);
    }

    public function create(array $request)
    {
        $rules = [
            'profile_id' => ['required', 'integer', 'exists:pr_profile,id'],
            'day_number' => [
                'required',
                'integer',
                Rule::unique('pr_daily_programs', 'day_number')->where(function ($q) use ($request) {
                    return $q->where('profile_id', $request['profile_id'] ?? null);
                }),
            ],
            'day_date' => [
                'nullable',
                'date',
                Rule::unique('pr_daily_programs', 'day_date')->where(function ($q) use ($request) {
                    return $q->where('profile_id', $request['profile_id'] ?? null);
                }),
            ],
        ];

        $validator = Validator::make($request, $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->repository->create($request);
    }

    public function update(array $request)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:pr_daily_programs,id'],
            'profile_id' => ['sometimes', 'integer', 'exists:pr_profile,id'],
            'day_number' => [
                'sometimes',
                'integer',
                Rule::unique('pr_daily_programs', 'day_number')
                    ->where(function ($q) use ($request) {
                        $profileId = $request['profile_id'] ?? null;
                        if (! $profileId && isset($request['id'])) {
                            $existing = DailyPrograms::find($request['id']);
                            $profileId = $existing ? $existing->profile_id : null;
                        }

                        return $q->where('profile_id', $profileId);
                    })
                    ->ignore($request['id'] ?? null),
            ],
            'day_date' => [
                'sometimes',
                'nullable',
                'date',
                Rule::unique('pr_daily_programs', 'day_date')
                    ->where(function ($q) use ($request) {
                        $profileId = $request['profile_id'] ?? null;
                        if (! $profileId && isset($request['id'])) {
                            $existing = DailyPrograms::find($request['id']);
                            $profileId = $existing ? $existing->profile_id : null;
                        }

                        return $q->where('profile_id', $profileId);
                    })
                    ->ignore($request['id'] ?? null),
            ],
        ];

        $validator = Validator::make($request, $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        $item = $this->repository->findByIdOrFail($request['id']);

        return $this->repository->update($item, $request);
    }
}
