<?php

namespace App\Http\Controllers;

use App\Models\Accounting\Invoice;
use App\Models\Accounting\UnpostedCollections;
use App\Models\General\Currency;
use App\Models\General\Service as GeneralService;
use App\Models\invoice\InvoiceServices;
use App\Models\Profile\Customer;
use App\Models\Profile\Profile;
use App\Models\Profile\ProfileTraveler;
use App\Models\Suppliers\Suppliers;
use App\Models\User;
use App\sys\ApiResponse;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class HomeController extends Controller
{
    use ApiResponse;

    public function dashboard()
    {
        $profileStatusCounts = $this->getProfileStatusCounts();
        $suppliers = $this->getSuppliersOutstanding();
        $customer = $this->getCustomersBalance();
        $data = [
            [
                'title' => 'dashboard.totalSuppliers',
                'description' => 'dashboard.totalSuppliersDesc',
                'values' => [[
                    'amount' => Suppliers::count(),
                    'currency' => null,
                ]],
                'icon' => 'ti ti-truck',
            ],
            [
                'title' => 'dashboard.totalCustomers',
                'description' => 'dashboard.totalCustomersDesc',
                'values' => [[
                    'amount' => Customer::count(),
                    'currency' => null,
                ]],
                'icon' => 'ti ti-users',
            ],
            [
                'title' => 'dashboard.supplierBalance',
                'description' => 'dashboard.supplierBalanceDesc',
                'values' => ! empty($suppliers) ? $suppliers : [['amount' => 0, 'currency' => null]],
                'icon' => 'ti ti-wallet',
            ],
            [
                'title' => 'dashboard.customerBalance',
                'description' => 'dashboard.customerBalanceDesc',
                'values' => $customer['total_travelers_amount'] ?? [['amount' => 0, 'currency' => null]],
                'icon' => 'ti ti-wallet-off',
            ], [
                'title' => 'dashboard.customerPayments',
                'description' => 'dashboard.customerPaymentsDesc',
                'values' => $customer['total_paid'] ?? [['amount' => 0, 'currency' => null]],
                'icon' => 'ti ti-cash-banknote-off',
            ], [
                'title' => 'dashboard.totalRemaining',
                'description' => 'dashboard.totalRemainingDesc',
                'values' => $customer['remaining'] ?? [['amount' => 0, 'currency' => null]],
                'icon' => 'ti ti-scale',
            ], [
                'title' => 'dashboard.openTrips',
                'description' => 'dashboard.openTripsDesc',
                'values' => [['amount' => $profileStatusCounts['followup'] ?? 0, 'currency' => null]],
                'icon' => 'ti ti-plane-departure',
            ], [
                'title' => 'dashboard.pendingPayments',
                'description' => 'dashboard.pendingPaymentsDesc',
                'values' => [['amount' => $profileStatusCounts['pending'] ?? 0, 'currency' => null]],
                'icon' => 'ti ti-clock-hour-4',
            ],
        ];

        return $this->apiResponse(200, 'success', null, $data);
    }

    public function profilesChartDashboard($period = 'week')
    {
        $allowedPeriods = ['week', 'month', 'threeMonths', 'year'];

        if (! in_array($period, $allowedPeriods, true)) {
            return $this->apiResponse(422, 'Invalid period', ['period' => 'Unsupported period value'], null);
        }

        [$startDate, $endDate] = $this->resolvePeriodRange($period);

        $statusVariants = [
            'pending' => ['pending'],
            'followup' => ['followup'],
            'done' => ['done'],
            'cancel' => ['cancel'],
        ];

        $rows = Profile::query()
            ->select(DB::raw('LOWER(status) as normalized_status'), DB::raw('COUNT(*) as total'))
            ->whereNotNull('status')
            ->whereBetween('created_at', [$startDate, $endDate])
            ->groupBy(DB::raw('LOWER(status)'))
            ->pluck('total', 'normalized_status');

        $counts = [];
        $totalProfiles = 0;

        foreach ($statusVariants as $label => $variants) {
            $count = 0;
            foreach ($variants as $variant) {
                $count += (int) ($rows[$variant] ?? 0);
            }
            $counts[$label] = $count;
            $totalProfiles += $count;
        }

        $chartData = [];
        foreach ($counts as $label => $count) {
            $chartData[] = [
                'status' => $label,
                'count' => $count,
                'percentage' => $totalProfiles > 0 ? round(($count / $totalProfiles) * 100, 2) : 0,
            ];
        }

        return $this->apiResponse(200, 'success', null, [
            'period' => $period,
            'start_date' => $startDate->toDateString(),
            'end_date' => $endDate->toDateString(),
            'data' => $chartData,
        ]);
    }

    public function dailyCheckInAndOutChartDashboard()
    {
        $startOfWeek = Carbon::now()->copy()->startOfWeek(Carbon::SUNDAY);
        $endOfWeek = $startOfWeek->copy()->addDays(6);

        $statusFilter = function ($query) {
            $query->whereNull('status')
                ->orWhereRaw('LOWER(status) != ?', ['cancel']);
        };

        $checkInCounts = Profile::query()
            ->select(DB::raw('DATE(arrival_date) as day'), DB::raw('COUNT(*) as total'))
            ->whereBetween('arrival_date', [$startOfWeek->toDateString(), $endOfWeek->toDateString()])
            ->whereNotNull('arrival_date')
            ->where($statusFilter)
            ->groupBy(DB::raw('DATE(arrival_date)'))
            ->pluck('total', 'day');

        $checkOutCounts = Profile::query()
            ->select(DB::raw('DATE(departure_date) as day'), DB::raw('COUNT(*) as total'))
            ->whereBetween('departure_date', [$startOfWeek->toDateString(), $endOfWeek->toDateString()])
            ->whereNotNull('departure_date')
            ->where($statusFilter)
            ->groupBy(DB::raw('DATE(departure_date)'))
            ->pluck('total', 'day');

        $categories = [];
        $checkInSeries = [];
        $checkOutSeries = [];

        for ($i = 0; $i < 7; $i++) {
            $currentDay = $startOfWeek->copy()->addDays($i);
            $dayKey = $currentDay->toDateString();
            $categories[] = $currentDay->format('l');
            $checkInSeries[] = (int) ($checkInCounts[$dayKey] ?? 0);
            $checkOutSeries[] = (int) ($checkOutCounts[$dayKey] ?? 0);
        }

        $data = [
            'series' => [
                [
                    'name' => 'Check-out',
                    'data' => $checkOutSeries,
                ],
                [
                    'name' => 'Check-in',
                    'data' => $checkInSeries,
                ],
            ],
            'categories' => $categories,
        ];

        return $this->apiResponse(200, 'success', null, $data);
    }

    public function invoicesAndPaymentsChartDashboard()
    {
        $year = Carbon::now()->year;
        $start = Carbon::create($year)->startOfYear();
        $end = Carbon::create($year)->endOfYear();

        $dataset = $this->buildInvoicesPaymentsDataset($start, $end, true);

        return $this->apiResponse(200, 'success', null, [
            'year' => $year,
            'series' => [
                [
                    'name' => 'Total Invoices',
                    'data' => $dataset['invoices'],
                ],
                [
                    'name' => 'Total Payments',
                    'data' => $dataset['payments'],
                ],
            ],
            'categories' => $dataset['categories'],
        ]);
    }

    public function monthlyInMonthlyInvoiceChartDashboard()
    {
        $end = Carbon::now()->endOfMonth();
        $start = $end->copy()->subMonths(5)->startOfMonth();

        $dataset = $this->buildInvoicesPaymentsDataset($start, $end, false);

        $totalInvoices = array_sum($dataset['invoices']);
        $currentMonthInvoices = end($dataset['invoices']) ?: 0;
        $previousMonthInvoices = count($dataset['invoices']) > 1 ? $dataset['invoices'][count($dataset['invoices']) - 2] : 0;
        $percentageChange = $previousMonthInvoices > 0
            ? round((($currentMonthInvoices - $previousMonthInvoices) / $previousMonthInvoices) * 100, 2)
            : 0;

        $allPaymentsCompleted = true;
        foreach ($dataset['invoices'] as $index => $invoiceAmount) {
            if (($dataset['payments'][$index] ?? 0) < $invoiceAmount) {
                $allPaymentsCompleted = false;
                break;
            }
        }

        $peakValue = max($dataset['invoices']);
        $peakIndex = array_search($peakValue, $dataset['invoices'], true);
        $peakCategory = $dataset['categories'][$peakIndex] ?? null;
        $peakPeriod = $dataset['periods'][$peakIndex] ?? null;
        $peakLabel = $peakPeriod
            ? sprintf('$%s (Invoices - %s)', number_format($peakValue, 2), $peakPeriod->format('F'))
            : null;

        return $this->apiResponse(200, 'success', null, [
            'totalInvoices' => round($totalInvoices, 2),
            'percentageChange' => $percentageChange,
            'allPaymentsCompleted' => $allPaymentsCompleted,
            'series' => [
                [
                    'name' => 'Invoices',
                    'data' => $dataset['invoices'],
                ],
                [
                    'name' => 'Payments',
                    'data' => $dataset['payments'],
                ],
            ],
            'categories' => $dataset['categories'],
            'peakPoint' => [
                'category' => $peakCategory,
                'value' => $peakValue,
                'label' => $peakLabel,
            ],
        ]);
    }

    public function salesPerformanceChartDashboard()
    {
        $statuses = ['followup', 'done'];
        $days = 60;
        $endDate = Carbon::now()->endOfDay();
        $startDate = $endDate->copy()->subDays($days - 1)->startOfDay();

        $defaultCurrencyRate = Currency::where('is_default', 1)->value('exchange_rate') ?? 1;
        $currencyRates = Currency::pluck('exchange_rate', 'id');

        $rows = ProfileTraveler::query()
            ->select(
                'profiles.user_id',
                DB::raw('DATE(profiles.created_at) as sale_day'),
                'pr_profile_travelers.currency_id',
                'pr_profile_travelers.total_price'
            )
            ->join('pr_profile as profiles', 'profiles.id', '=', 'pr_profile_travelers.profile_id')
            ->whereNotNull('profiles.user_id')
            ->whereBetween('profiles.created_at', [$startDate->toDateString(), $endDate->toDateString()])
            ->whereIn(DB::raw('LOWER(profiles.status)'), $statuses)
            ->get();

        $userTotals = [];
        $userDaily = [];

        foreach ($rows as $row) {
            $currencyRate = $currencyRates[$row->currency_id] ?? $defaultCurrencyRate;
            $amount = (float) $row->total_price * ($currencyRate / $defaultCurrencyRate);
            $dayKey = $row->sale_day;

            $userTotals[$row->user_id] = ($userTotals[$row->user_id] ?? 0) + $amount;
            $userDaily[$row->user_id][$dayKey] = ($userDaily[$row->user_id][$dayKey] ?? 0) + $amount;
        }

        arsort($userTotals);
        $topUserIds = array_slice(array_keys($userTotals), 0, 5);

        if (empty($topUserIds)) {
            return $this->apiResponse(200, 'success', null, [
                'employees' => [],
                'selectedEmployeeId' => null,
            ]);
        }

        $users = User::whereIn('id', $topUserIds)->get(['id', 'name'])->keyBy('id');

        $dateKeys = [];
        for ($index = 0; $index < $days; $index++) {
            $dateKeys[] = $startDate->copy()->addDays($index)->toDateString();
        }

        $employees = [];

        foreach ($topUserIds as $userId) {
            $salesData = [];
            foreach ($dateKeys as $dayKey) {
                $salesData[] = round($userDaily[$userId][$dayKey] ?? 0, 2);
            }
            $peakValue = max($salesData);
            $peakIndex = $peakValue > 0 ? array_search($peakValue, $salesData, true) : null;

            $employees[] = [
                'id' => $userId,
                'name' => $users[$userId]->name ?? 'Unknown',
                'salesData' => $salesData,
                'peakPerformance' => [
                    'index' => $peakIndex,
                    'value' => round($peakValue, 2),
                ],
            ];
        }

        return $this->apiResponse(200, 'success', null, [
            'employees' => $employees,
            'selectedEmployeeId' => $topUserIds[0] ?? null,
        ]);
    }

    public function invoiceServices()
    {
        $rows = InvoiceServices::query()
            ->select('service_id', DB::raw('SUM(grand_total) as total_revenue'))
            ->whereNotNull('service_id')
            ->where('travel_tourism_type', '!=', 'accommodation')
            ->groupBy('service_id')
            ->orderByDesc('total_revenue')
            ->limit(5)
            ->get();

        if ($rows->isEmpty()) {
            return $this->apiResponse(200, 'success', null, []);
        }

        $serviceIds = $rows->pluck('service_id')->filter()->unique();
        $services = GeneralService::with('currentTranslation')
            ->whereIn('id', $serviceIds)
            ->get()
            ->keyBy('id');

        $palette = ['#5B4FFF', '#38BDF8', '#9CA3AF', '#F97316', '#22C55E'];
        $response = [];

        foreach ($rows as $index => $row) {
            $service = $services->get($row->service_id);
            $name = $service?->currentTranslation?->title ?? $service?->title ?? null;

            $response[] = [
                'name' => $name,
                'value' => round((float) $row->total_revenue, 2),
                'color' => $palette[$index % count($palette)],
            ];
        }

        return $this->apiResponse(200, 'success', null, $response);
    }

    private function getSuppliersOutstanding(): array
    {
        $rows = InvoiceServices::query()
            ->select('currency_id', DB::raw('SUM(remaining_to_supplier) as total_remaining'))
            ->where('is_paid', 0)
            ->whereNotNull('currency_id')
            ->groupBy('currency_id')
            ->get();

        if ($rows->isEmpty()) {
            return [];
        }

        $currencies = Currency::whereIn('id', $rows->pluck('currency_id'))
            ->get()
            ->keyBy('id');

        return $rows->map(function ($row) use ($currencies) {
            $currency = $currencies->get($row->currency_id);

            return [
                'currency' => $currency->code ?? null,
                'amount' => round((float) $row->total_remaining, 2),
            ];
        })->values()->toArray();
    }

    private function getCustomersBalance(): array
    {
        $excludedStatuses = ['cancel', 'done'];
        $normalizedStatuses = array_map('strtolower', $excludedStatuses);

        $travelerTotals = ProfileTraveler::query()
            ->select('pr_profile_travelers.currency_id', DB::raw('SUM(pr_profile_travelers.total_price) as total_amount'))
            ->join('pr_profile as profiles', 'profiles.id', '=', 'pr_profile_travelers.profile_id')
            ->whereNull('profiles.deleted_at')
            ->whereNotNull('pr_profile_travelers.currency_id')
            ->where(function ($query) use ($normalizedStatuses) {
                $query->whereNull('profiles.status')
                    ->orWhereNotIn(DB::raw('LOWER(profiles.status)'), $normalizedStatuses);
            })
            ->groupBy('pr_profile_travelers.currency_id')
            ->pluck('total_amount', 'currency_id');

        $collectionsTotals = UnpostedCollections::query()
            ->select('unposted_collections.currency_id', DB::raw('SUM(unposted_collections.amount) as total_paid'))
            ->join('pr_profile as profiles', 'profiles.id', '=', 'unposted_collections.profile_id')
            ->whereNull('profiles.deleted_at')
            ->whereNotNull('unposted_collections.currency_id')
            ->where(function ($query) use ($normalizedStatuses) {
                $query->whereNull('profiles.status')
                    ->orWhereNotIn(DB::raw('LOWER(profiles.status)'), $normalizedStatuses);
            })
            ->groupBy('unposted_collections.currency_id')
            ->pluck('total_paid', 'currency_id');

        $currencyIds = $travelerTotals->keys()
            ->merge($collectionsTotals->keys())
            ->filter()
            ->unique();

        if ($currencyIds->isEmpty()) {
            return [];
        }

        $currencies = Currency::whereIn('id', $currencyIds)->get()->keyBy('id');

        $balances = [];

        foreach ($currencyIds as $currencyId) {
            $currency = $currencies->get($currencyId);
            $totalTravelers = round((float) ($travelerTotals[$currencyId] ?? 0), 2);
            $totalPaid = round((float) ($collectionsTotals[$currencyId] ?? 0), 2);
            $remaining = round($totalTravelers - $totalPaid, 2);

            $balances['total_paid'][] = [
                'currency' => $currency->code ?? null,
                'amount' => $totalPaid,
            ];
            $balances['total_travelers_amount'][] = [
                'currency' => $currency->code ?? null,
                'amount' => $totalTravelers,
            ];
            $balances['remaining'][] = [
                'currency' => $currency->code ?? null,
                'amount' => $remaining,
            ];
        }

        return $balances;
    }

    private function getProfileStatusCounts(): array
    {
        $targetStatuses = ['followup', 'pending'];

        $rows = Profile::query()
            ->select(DB::raw('LOWER(status) as normalized_status'), DB::raw('COUNT(*) as total'))
            ->whereNotNull('status')
            ->whereIn(DB::raw('LOWER(status)'), $targetStatuses)
            ->groupBy(DB::raw('LOWER(status)'))
            ->pluck('total', 'normalized_status');

        $counts = [];
        foreach ($targetStatuses as $status) {
            $counts[$status] = (int) ($rows[$status] ?? 0);
        }

        return $counts;
    }

    private function buildInvoicesPaymentsDataset(Carbon $start, Carbon $end, bool $useFullMonthNames = true): array
    {
        $rows = Invoice::query()
            ->selectRaw('YEAR(invoice_date) as year')
            ->selectRaw('MONTH(invoice_date) as month')
            ->selectRaw('SUM(invoice_price) as total_invoices')
            ->selectRaw('SUM(paid_price) as total_payments')
            ->whereNotNull('invoice_date')
            ->whereBetween('invoice_date', [$start->toDateString(), $end->toDateString()])
            ->groupBy('year', 'month')
            ->get()
            ->keyBy(function ($row) {
                return sprintf('%04d-%02d', $row->year, $row->month);
            });

        $categories = [];
        $invoices = [];
        $payments = [];
        $periods = [];

        $current = $start->copy();
        while ($current <= $end) {
            $key = sprintf('%04d-%02d', $current->year, $current->month);
            $row = $rows->get($key);

            $categories[] = $useFullMonthNames ? $current->format('F') : strtoupper($current->format('M'));
            $invoices[] = $row ? round((float) $row->total_invoices, 2) : 0;
            $payments[] = $row ? round((float) $row->total_payments, 2) : 0;
            $periods[] = $current->copy();

            $current->addMonth();
        }

        return [
            'categories' => $categories,
            'invoices' => $invoices,
            'payments' => $payments,
            'periods' => $periods,
        ];
    }

    private function resolvePeriodRange(string $period): array
    {
        $now = Carbon::now();

        return match ($period) {
            'week' => [$now->copy()->startOfWeek(), $now],
            'month' => [$now->copy()->startOfMonth(), $now],
            'threeMonths' => [$now->copy()->subMonths(3), $now],
            'year' => [$now->copy()->startOfYear(), $now],
            default => [$now->copy()->startOfWeek(), $now],
        };
    }
}
