<?php

namespace App\Http\Controllers\Invoice;

use App\Http\Controllers\Controller;
use App\Http\Resources\Invoice\InvoiceServicesCollection;
use App\Http\Resources\Invoice\InvoiceServicesResource;
use App\sys\ApiResponse;
use App\sys\Services\Invoice\InvoiceServicesService;
use Illuminate\Http\Request;

class InvoiceServicesController extends Controller
{
    use ApiResponse;

    protected InvoiceServicesService $invoiceServicesService;

    public function __construct(InvoiceServicesService $invoiceServicesService)
    {
        $this->invoiceServicesService = $invoiceServicesService;
    }

    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $filters = $request->only(['profile_id', 'travel_tourism_type', 'service_id', 'invoice_id', 'limit']);

        $invoiceServices = $this->invoiceServicesService->getList($filters);

        if (! $invoiceServices) {
            $errors = $this->invoiceServicesService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, new InvoiceServicesCollection($invoiceServices));
    }

    public function getHandlingServices(Request $request)
    {
        $filters = $request->only(['profile_id', 'travel_tourism_type', 'service_id', 'invoice_id', 'limit', 'executive_id', 'is_by_handling']);

        $invoiceServices = $this->invoiceServicesService->getHandlingServices($filters);

        if (! $invoiceServices) {
            $errors = $this->invoiceServicesService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, new InvoiceServicesCollection($invoiceServices));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $response = $this->invoiceServicesService->create($request->all());

        if (! $response) {
            $errors = $this->invoiceServicesService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        $serviceType = $request->input('travel_tourism_type', 'accommodation');

        return $this->apiResponse(200, 'success', null, new InvoiceServicesResource($response, $serviceType));
    }

    /**
     * Display the specified resource.
     */
    public function show(int $id)
    {
        $invoiceService = $this->invoiceServicesService->getById($id);

        if (! $invoiceService) {
            $errors = $this->invoiceServicesService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        $serviceType = $invoiceService->travel_tourism_type ?? 'accommodation';

        return $this->apiResponse(200, 'success', null, new InvoiceServicesResource($invoiceService, $serviceType));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request)
    {

        $response = $this->invoiceServicesService->update($request->all());

        if (! $response) {
            $errors = $this->invoiceServicesService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        $serviceType = $request->input('travel_tourism_type', $response->travel_tourism_type);

        return $this->apiResponse(200, 'success', null, new InvoiceServicesResource($response, $serviceType));
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Request $request)
    {
        $result = $this->invoiceServicesService->deleteMany($request->input('ids'));

        if ($result === false) {
            $errors = $this->invoiceServicesService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, $result);
    }

    /**
     * Get invoice services by profile ID
     */
    public function getByProfile(int $profileId, $type)
    {
        $invoiceServices = $this->invoiceServicesService->getByProfile($profileId, $type);

        return $this->apiResponse(200, 'success', null, new InvoiceServicesCollection($invoiceServices, $type));
    }

    public function getByDaily(int $daily_id, $type)
    {
        $invoiceServices = $this->invoiceServicesService->getByDaily($daily_id, $type);

        return $this->apiResponse(200, 'success', null, new InvoiceServicesCollection($invoiceServices, $type));
    }

    /**
     * Get validation rules for a specific service type
     */
    public function getValidationRules(Request $request, string $serviceType)
    {
        $rules = $this->invoiceServicesService->getValidationRulesForType($serviceType);

        if (empty($rules)) {
            return $this->apiResponse(400, 'fail', ['service_type' => ['Unsupported service type']], null);
        }

        return $this->apiResponse(200, 'success', null, [
            'service_type' => $serviceType,
            'validation_rules' => $rules,
        ]);
    }

    /**
     * Get supported service types
     */
    public function getSupportedTypes()
    {
        $types = $this->invoiceServicesService->getSupportedServiceTypes();

        return $this->apiResponse(200, 'success', null, [
            'supported_types' => $types,
        ]);
    }

    /**
     * Get all audit changes for an invoice service
     */
    public function getAllChanges(int $id)
    {
        $changes = $this->invoiceServicesService->getAllChanges($id);

        if ($changes === false) {
            $errors = $this->invoiceServicesService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, $changes);
    }

    public function search()
    {
        $results = $this->invoiceServicesService->search();

        return $this->apiResponse(200, 'success', null, $results);
    }

    public function paidSuppliers(Request $request)
    {
        if ($this->invoiceServicesService->paidSuppliers($request->all())) {
            return $this->apiResponse(200, 'success', null, null);
        }
        $errors = $this->invoiceServicesService->errors();

        return $this->apiResponse(400, 'fail', $errors, null);
    }

    public function getsummary($profileId)
    {

        $summary = $this->invoiceServicesService->GetSummary($profileId);

        if ($summary === false) {
            $errors = $this->invoiceServicesService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, $summary);

    }

    /**
     * Update invoice service status (pending, confirmed, cancelled, done)
     */
    public function status(Request $request)
    {
        $response = $this->invoiceServicesService->updateStatus($request->all());
        if (! $response) {
            $errors = $this->invoiceServicesService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, new InvoiceServicesResource($response));
    }
}
