<?php

namespace App\Http\Controllers\Notification;

use App\Http\Controllers\Controller;
use App\Http\Resources\Notification\NotificationCollection;
use App\Http\Resources\Notification\NotificationResource;
use App\sys\ApiResponse;
use App\sys\Services\Notification\NotificationService;
use Illuminate\Http\Request;

class NotificationController extends Controller
{
    use ApiResponse;

    protected NotificationService $notificationService;

    public function __construct(NotificationService $notificationService)
    {
        $this->notificationService = $notificationService;
    }

    /**
     * Get all notifications for authenticated user
     */
    public function index(Request $request)
    {
        $userId = auth('api')->id();
        $limit = $request->get('limit', 15);
        $unreadOnly = false;

        $notifications = $this->notificationService->getUserNotifications($userId, $limit, $unreadOnly);

        return $this->apiResponse(200, 'success', null, new NotificationCollection($notifications));
    }

    /**
     * Get unread notifications only
     */
    public function unread(Request $request)
    {
        $userId = auth('api')->id();
        $limit = $request->get('limit', 15);

        $notifications = $this->notificationService->getUserNotifications($userId, $limit, true);

        return $this->apiResponse(200, 'success', null, new NotificationCollection($notifications));
    }

    /**
     * Get unread count
     */
    public function unreadCount()
    {
        $userId = auth('api')->id();
        $count = $this->notificationService->getUnreadCount($userId);

        return $this->apiResponse(200, 'success', null, ['count' => $count]);
    }

    /**
     * Mark notification as read
     */
    public function markAsRead($id)
    {
        $userId = auth('api')->id();
        $result = $this->notificationService->markAsRead($id, $userId);

        if (! $result) {
            return $this->apiResponse(404, 'fail', ['notification' => ['Notification not found']], null);
        }

        return $this->apiResponse(200, 'success', null, null);
    }

    /**
     * Mark all notifications as read
     */
    public function markAllAsRead()
    {
        $userId = auth('api')->id();
        $count = $this->notificationService->markAllAsRead($userId);

        return $this->apiResponse(200, 'success', null, ['marked_count' => $count]);
    }

    /**
     * Delete a notification
     */
    public function destroy($id)
    {
        $userId = auth('api')->id();
        $result = $this->notificationService->delete($id, $userId);

        if (! $result) {
            return $this->apiResponse(404, 'fail', ['notification' => ['Notification not found']], null);
        }

        return $this->apiResponse(200, 'success', null, null);
    }

    /**
     * Get single notification
     */
    public function show($id)
    {
        $userId = auth('api')->id();
        $notification = $this->notificationService->find($id, $userId);

        if (! $notification) {
            return $this->apiResponse(404, 'fail', ['notification' => ['Notification not found']], null);
        }

        return $this->apiResponse(200, 'success', null, new NotificationResource($notification));
    }
}
