<?php

namespace App\Http\Controllers\Profile;

use App\Http\Controllers\Controller;
use App\Http\Resources\Profile\CustomerCollection;
use App\Http\Resources\Profile\CustomerHistoryResource;
use App\Http\Resources\Profile\CustomerResource;
use App\sys\ApiResponse;
use App\sys\Services\Profile\CustomerService;
use Illuminate\Http\Request;

class CustomerController extends Controller
{
    use ApiResponse;

    protected CustomerService $customerService;

    public function __construct(CustomerService $customerService)
    {
        $this->customerService = $customerService;
    }

    public function index(Request $request)
    {
        $items = $this->customerService->getPaginated($request->all());

        return $this->apiResponse(200, 'success', null, new CustomerCollection($items));
    }

    public function show($id)
    {
        $item = $this->customerService->getById($id);
        if (! $item) {
            $errors = $this->customerService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, new CustomerResource($item));
    }

    public function store(Request $request)
    {
        $response = $this->customerService->create($request->all());
        if (! $response) {
            $errors = $this->customerService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, new CustomerResource($response));
    }

    public function update(Request $request)
    {
        $response = $this->customerService->update($request->all());
        if (! $response) {
            $errors = $this->customerService->errors();

            return $this->apiResponse(400, 'fail', null, $errors);
        }

        return $this->apiResponse(200, 'success', null, new CustomerResource($response));
    }

    public function destroy(Request $request)
    {
        $ids = $request->input('ids', []);
        if (! is_array($ids) || empty($ids)) {
            return $this->apiResponse(400, 'No IDs provided for deletion', null, null);
        }
        $result = $this->customerService->del($ids);
        if ($result) {
            return $this->apiResponse(200, 'Customers deleted successfully', null, null);
        }
        $errors = $this->customerService->errors();

        return $this->apiResponse(400, 'fail', $errors, null);
    }

    public function getActive()
    {
        $data = $this->customerService->getActive();
        $collect = new CustomerCollection($data);

        return $this->apiResponse(200, 'Active Customers retrieved successfully', null, $collect);
    }

    public function getAllChanges($id)
    {
        $customer = $this->customerService->getById($id);
        if (! $customer) {
            $errors = $this->customerService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        $changes = $this->customerService->getAllChanges($customer);

        return $this->apiResponse(200, 'success', null, CustomerHistoryResource::collection($changes));
    }
}
