<?php

namespace App\sys\Repository\General;

use App\Models\General\Domain;
use App\Models\General\TaxRate;

class DomainRepository
{
    private $columns = [
        'id' => 'id',
        'name' => 'title',
        'type' => 'type',
        'tax_rate_id' => 'tax_rate_id',
        'active' => 'enabled_disable',
        'created_at' => 'created',
        'updated_at' => 'modified',
    ];

    public function getPaginated()
    {
        $column = request('sort_by', null);
        $order = request('sort_order', 'asc');
        $title = request('name', null);
        $limit = request('limit', 15);

        return Domain::when($title, function ($q, $title) {
            $q->whereHas('currentTranslation', function ($q) use ($title) {
                $q->where('title', 'LIKE', "%$title%");
            });
        })->when($column && array_key_exists($column, $this->columns), function ($query) use ($column, $order) {
            if ($column == 'title') {
                $query->leftJoin('domain_translations', function ($join) {
                    $join->on('domains.id', '=', 'domain_translations.domain_id')
                        ->where('domain_translations.lang_id', app('lang_id'));
                })->orderBy('domain_translations.title', $order)->select('domains.*');
            } else {
                $query->orderBy($this->columns[$column], $order);
            }
        })->with('currentTranslation', 'taxRate')->paginate($limit);
    }

    public function findByIdOrFail(int $id)
    {
        return Domain::with('currentTranslation', 'taxRate')->find($id);
    }

    public function create(array $data)
    {
        $domain = new Domain;
        $domain->title = $data['name'];
        $domain->tax_rate_id = $data['tax_rate_id'] ?? null;
        $domain->type = $data['type'] ?? null;
        $domain->enabled_disable = $data['active'];
        $domain->save();

        return $domain;
    }

    public function update(Domain $domain, array $data)
    {
        $domain->title = $data['name'] ?? $domain->title;
        $domain->tax_rate_id = $data['tax_rate_id'] ?? $domain->tax_rate_id;
        $domain->type = $data['type'] ?? $domain->type;
        $domain->enabled_disable = $data['active'] ?? $domain->enabled_disable;
        $domain->save();

        return $domain;
    }

    public function delete(Domain $domain)
    {
        return $domain->delete();
    }

    public function del(array $ids)
    {
        return Domain::whereIn('id', $ids)->delete();
    }

    public function getActive()
    {
        return Domain::with('currentTranslation', 'taxRate')->where('enabled_disable', 1)->get();
    }

    public function getByIdWithTranslation($id)
    {
        return Domain::with('translations', 'taxRate')->find($id);
    }

    public function getByTaxRateId(int $taxRateId)
    {
        return Domain::with('currentTranslation', 'taxRate')
            ->where('tax_rate_id', $taxRateId)
            ->get();
    }

    public function getAllChanges($domain)
    {
        return $domain->audits()
            ->with(['user' => function ($query) {
                $query->select('id', 'name');
            }])
            ->get()
            ->map(function ($audit) {
                $oldValues = $audit->old_values ?? [];
                $newValues = $audit->new_values ?? [];

                $taxRateIds = array_filter([
                    $oldValues['tax_rate_id'] ?? null,
                    $newValues['tax_rate_id'] ?? null,
                ]);

                $taxRates = $taxRateIds ? TaxRate::whereIn('id', $taxRateIds)
                    ->pluck('title', 'id')
                    ->toArray() : [];

                if (isset($oldValues['tax_rate_id']) && isset($taxRates[$oldValues['tax_rate_id']])) {
                    $oldValues['tax_rate_title'] = $taxRates[$oldValues['tax_rate_id']];
                }
                if (isset($newValues['tax_rate_id']) && isset($taxRates[$newValues['tax_rate_id']])) {
                    $newValues['tax_rate_title'] = $taxRates[$newValues['tax_rate_id']];
                }

                return [
                    'audit_id' => $audit->id,
                    'user_id' => $audit->user_id ?? null,
                    'user' => $audit->user ? $audit->user->toArray() : null,
                    'old_values' => $oldValues,
                    'new_values' => $newValues,
                    'changed_at' => $audit->created_at,
                    'event' => $audit->event,
                    'ip_address' => $audit->ip_address,
                    'user_agent' => $audit->user_agent,
                ];
            })
            ->values();
    }
}
