<?php

namespace App\sys\Repository\General;

use App\Models\Accounting\TreeAccounting;
use App\Models\General\Currency;
use App\Models\General\TaxRate;
use App\Models\General\TaxRateMappings;

class TaxRateRepository
{
    public function create(array $data)
    {
        $taxRate = new TaxRate;
        $taxRate->title = $data['name'];
        $taxRate->percentage = $data['percentage'] ?? null;
        $taxRate->type = $data['type'] ?? null;
        $taxRate->is_active = $data['active'];
        $taxRate->save();
        $this->saveAccounting($data, $taxRate->id);

        return $taxRate->load('accounting');
    }

    public function findByIdOrFail(int $id): TaxRate
    {
        return TaxRate::with('accounting')->find($id);
    }

    public function getTaxsByCurrency($currency)
    {
        return TaxRate::with('currentTranslation')->whereHas('accounting', function ($query) use ($currency) {
            $query->where('currency_id', $currency);
        })->where('is_active', 1)->get();
    }

    public function getByIdWithTranslation(int $id): ?TaxRate
    {
        return TaxRate::with(['translations' => function ($q) {
            $q->with('getLang');
        }])->find($id);
    }

    public function getPaginated()
    {
        $column = request('sort_by', null);
        $order = request('sort_order', 'asc');
        $name = request('name', null);
        $limit = request('limit', 15);

        return TaxRate::when($name, function ($q, $name) {
            $q->whereHas('currentTranslation', function ($q) use ($name) {
                $q->where('title', 'LIKE', "%$name%");
            });
        })->when($column && in_array($column, ['id', 'percentage', 'type', 'is_active', 'created', 'modified']), function ($query) use ($column, $order) {
            $query->orderBy($column, $order);
        })->with('currentTranslation')->paginate($limit);
    }

    public function update(TaxRate $taxRate, array $data)
    {
        $taxRate->title = $data['name'] ?? $taxRate->title;
        $taxRate->percentage = $data['percentage'] ?? $taxRate->percentage;
        $taxRate->type = $data['type'] ?? $taxRate->type;
        $taxRate->is_active = $data['active'] ?? $taxRate->is_active;
        $taxRate->save();
        $this->updatedAccounting($data, $taxRate->id);

        return $taxRate->load('accounting');
    }

    public function delete(TaxRate $taxRate)
    {
        return $taxRate->delete();
    }

    public function del(array $ids)
    {
        return TaxRate::whereIn('id', $ids)->delete();
    }

    public function getActive()
    {
        return TaxRate::with('currentTranslation')->where('is_active', 1)->get();
    }

    private function updatedAccounting($data, $id)
    {
        foreach ($data['accounting'] ?? [] as $value) {
            if (TaxRateMappings::where(['tax_rate_id' => $id, 'type' => $value['input_name'], 'currency_id' => $value['currency_id']])->exists()) {
                $this->updatethisAccount($value, $id);
            } else {
                $this->addAccounting($value, $id);
            }
        }
    }

    public function updatethisAccount($data, $tax_rate_id)
    {
        $account = TaxRateMappings::where(['tax_rate_id' => $tax_rate_id, 'type' => $data['input_name'], 'currency_id' => $data['currency_id']])->first();
        $account->tree_account_id = $data['tree_account_id'];
        $account->save();
    }

    public function saveAccounting($data, $tax_rate_id)
    {
        foreach ($data['accounting'] as $value) {
            $this->addAccounting($value, $tax_rate_id);
        }
    }

    public function addAccounting($data, $tax_rate_id)
    {
        $account = new TaxRateMappings;
        $account->tax_rate_id = $tax_rate_id;
        $account->currency_id = $data['currency_id'];
        $account->tree_account_id = $data['tree_account_id'];
        $account->type = $data['input_name'];
        $account->save();
    }

    public function getAllChanges($taxRate)
    {
        // Get tax rate changes
        $taxRateChanges = $taxRate->audits()
            ->with(['user' => function ($query) {
                $query->select('id', 'name');
            }])
            ->get()
            ->map(function ($audit) {
                $oldValues = $audit->old_values ?? [];
                $newValues = $audit->new_values ?? [];

                return [
                    'audit_id' => $audit->id,
                    'user_id' => $audit->user_id ?? null,
                    'user' => $audit->user ? $audit->user->toArray() : null,
                    'old_values' => $oldValues,
                    'new_values' => $newValues,
                    'changed_at' => $audit->created_at,
                    'event' => $audit->event,
                    'ip_address' => $audit->ip_address,
                    'user_agent' => $audit->user_agent,
                    'audit_type' => 'tax_rate',
                ];
            });

        // Get accounting mapping changes
        $accountingChanges = TaxRateMappings::where('tax_rate_id', $taxRate->id)
            ->with(['audits' => function ($query) {
                $query->with(['user' => function ($userQuery) {
                    $userQuery->select('id', 'name');
                }]);
            }])
            ->get()
            ->flatMap(function ($accountingMapping) {
                return $accountingMapping->audits->map(function ($audit) {
                    $oldValues = $audit->old_values ?? [];
                    $newValues = $audit->new_values ?? [];

                    // Resolve currency and tree account names
                    $currencyIds = array_filter([
                        $oldValues['currency_id'] ?? null,
                        $newValues['currency_id'] ?? null,
                    ]);
                    $treeAccountIds = array_filter([
                        $oldValues['tree_account_id'] ?? null,
                        $newValues['tree_account_id'] ?? null,
                    ]);

                    $currencies = $currencyIds ? Currency::whereIn('id', $currencyIds)
                        ->pluck('name', 'id')
                        ->toArray() : [];
                    $treeAccounts = $treeAccountIds ? TreeAccounting::whereIn('tree_accounting_id', $treeAccountIds)
                        ->pluck('title', 'tree_accounting_id')
                        ->toArray() : [];

                    if (isset($oldValues['currency_id']) && isset($currencies[$oldValues['currency_id']])) {
                        $oldValues['currency_name'] = $currencies[$oldValues['currency_id']];
                    }
                    if (isset($newValues['currency_id']) && isset($currencies[$newValues['currency_id']])) {
                        $newValues['currency_name'] = $currencies[$newValues['currency_id']];
                    }

                    if (isset($oldValues['tree_account_id']) && isset($treeAccounts[$oldValues['tree_account_id']])) {
                        $oldValues['tree_account_name'] = $treeAccounts[$oldValues['tree_account_id']];
                    }
                    if (isset($newValues['tree_account_id']) && isset($treeAccounts[$newValues['tree_account_id']])) {
                        $newValues['tree_account_name'] = $treeAccounts[$newValues['tree_account_id']];
                    }

                    return [
                        'audit_id' => $audit->id,
                        'user_id' => $audit->user_id ?? null,
                        'user' => $audit->user ? $audit->user->toArray() : null,
                        'old_values' => $oldValues,
                        'new_values' => $newValues,
                        'changed_at' => $audit->created_at,
                        'event' => $audit->event,
                        'ip_address' => $audit->ip_address,
                        'user_agent' => $audit->user_agent,
                        'audit_type' => 'accounting_mapping',
                        'accounting_mapping_id' => $audit->auditable_id,
                    ];
                });
            });

        // Combine and sort all changes by timestamp
        $allChanges = $taxRateChanges->concat($accountingChanges)
            ->sortByDesc('changed_at')
            ->values();

        return $allChanges;
    }
}
