<?php

namespace App\sys\Services\Auth;

use App\sys\Repository\Auth\AuthRepository;
use App\sys\Services;
use Illuminate\Support\Facades\Validator;

class AuthService extends Services
{
    // مدة الـ Access Token بالدقائق (ساعة واحدة)
    private const ACCESS_TOKEN_TTL = 30000;

    // مدة الـ Refresh Token بالدقائق (أسبوعين)
    private const REFRESH_TOKEN_TTL = 20160;

    protected $authrepo;

    public function __construct(AuthRepository $authrepo)
    {
        $this->authrepo = $authrepo;
    }

    public function login(array $data)
    {
        $rules = [
            'email' => 'required|email',
            'password' => 'required|string',
        ];

        $validator = Validator::make($data, $rules);

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        $credentials = [
            'email' => $data['email'],
            'password' => $data['password'],
        ];

        // ✅ JWT Attempt
        if (! $token = auth('api')->attempt($credentials)) {
            $this->setError(['email' => ['Invalid credentials.']]);

            return false;
        }

        $user = auth('api')->user();

        if (! $user->is_active) {
            $this->setError(['email' => ['This account is inactive.']]);

            return false;
        }

        $user->load(
            'role',
            'groups',
            'services',
            'accommodations',
            'companies',
            'account_mappings.currency',
            'account_mappings.commissionAccount',
            'account_mappings.walletAccount'
        );

        // 2. إنشاء التوكنات
        $tokens = $this->generateTokens($user);

        return [
            'user' => $user,
            ...$tokens,
        ];
    }

    public function logout()
    {
        auth('api')->logout();

        return true;
    }

    // Method For Refersh Token
    public function refreshToken($refreshToken)
    {
        try {
            // 1. فك التوكن لقراءة البيانات
            $payload = auth('api')->setToken($refreshToken)->getPayload();

            // 2. التحقق أن هذا التوكن هو فعلاً Refresh Token
            if ($payload->get('type') !== 'refresh') {
                $this->setError(['token' => ['Invalid token type. Expected refresh token.']]);

                return false;
            }

            // 3. الحصول على المستخدم
            $userId = $payload->get('sub');
            // We can use the repository or model directly here, assuming User model is available
            $user = \App\Models\User::find($userId);

            if (! $user) {
                $this->setError(['user' => ['User not found']]);

                return false;
            }

            // 4. حرق (Invalidate) التوكن القديم
            auth('api')->setToken($refreshToken)->invalidate();

            // 5. إصدار توكنات جديدة تماماً
            return $this->generateTokens($user);
        } catch (\Tymon\JWTAuth\Exceptions\TokenBlacklistedException $e) {
            $this->setError(['token' => ['Token has been used already. Please login again.']]);

            return false;
        } catch (\Tymon\JWTAuth\Exceptions\JWTException $e) {
            $this->setError(['token' => ['Invalid or expired refresh token']]);

            return false;
        }
    }

    /**
     * دالة مساعدة لإنشاء التوكنين لمنع تكرار الكود
     */
    private function generateTokens($user)
    {
        // إنشاء Access Token
        $accessToken = auth('api')
            ->claims(['type' => 'access']) // علامة مميزة
            ->setTTL(self::ACCESS_TOKEN_TTL)
            ->fromUser($user);

        // إنشاء Refresh Token
        $refreshToken = auth('api')
            ->claims(['type' => 'refresh']) // علامة مميزة
            ->setTTL(self::REFRESH_TOKEN_TTL)
            ->fromUser($user);

        return [
            'access_token' => $accessToken,
            'refresh_token' => $refreshToken,
            'token_type' => 'Bearer',
            'expires_in' => self::ACCESS_TOKEN_TTL * 60, // بالثواني للفرونت إند
        ];
    }

    public function changePassword(array $data)
    {
        // 1. التحقق من المدخلات (Validation)
        $validator = Validator::make($data, [
            'old_password' => 'required|string',
            'new_password' => 'required|string|min:8|confirmed',
        ]);

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }
        $user = auth('api')->user();

        $data = $this->authrepo->changePassword($user, $data);

        auth('api')->logout();

        return $data;
    }
}
