<?php

namespace App\sys\Services\General;

use App\sys\Repository\General\AreaRepository;
use App\sys\Services;
use Illuminate\Support\Facades\Validator;

class AreaService extends Services
{
    protected AreaRepository $areaRepository;

    public function __construct(AreaRepository $areaRepository)
    {
        $this->areaRepository = $areaRepository;
    }

    public function getPaginatedAreas()
    {
        return $this->areaRepository->getPaginated();
    }

    public function getAreaById(int $id)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:area,id'],
        ];

        $validator = Validator::make(['id' => $id], $rules);

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->areaRepository->findByIdOrFail($id);
    }

    public function createArea($request)
    {
        $rules = [
            'city_id' => ['required', 'integer', 'exists:cities,id'],
            'name' => ['required', 'string', 'max:255'],
            'active' => ['required', 'in:0,1'],
        ];

        $validator = Validator::make($request, $rules);

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        $area = $this->areaRepository->create($request);
        $this->syncTranslations($area);

        return $area;
    }

    public function updateArea($request)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:area,id'],
            'city_id' => ['sometimes', 'integer', 'exists:cities,id'],
            'name' => ['sometimes', 'string', 'max:255'],
            'active' => ['sometimes', 'in:0,1'],
        ];

        $validator = Validator::make($request, $rules);

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        $area = $this->areaRepository->findByIdOrFail($request['id']);

        return $this->areaRepository->update($area, $request);
    }

    public function deleteArea(int $id)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:area,id'],
        ];

        $validator = Validator::make(['id' => $id], $rules);

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        $area = $this->areaRepository->findByIdOrFail($id);

        return $this->areaRepository->delete($area);
    }

    public function del(array $ids)
    {
        return $this->areaRepository->del($ids);
    }

    public function getActive()
    {
        return $this->areaRepository->getActive();
    }

    public function getByIdWithTranslation($id)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:area,id'],
        ];
        $validator = Validator::make(['id' => $id], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }
        $area = $this->areaRepository->getByIdWithTranslation($id);

        return $this->syncTranslations($area, true);
    }

    public function getAreasByCity($cityId)
    {
        $rules = [
            'city_id' => ['required', 'integer', 'exists:cities,id'],
        ];
        $validator = Validator::make(['city_id' => $cityId], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->areaRepository->getAreasByCity($cityId);
    }

    public function getAllChanges($area)
    {
        return $this->areaRepository->getAllChanges($area);
    }
}
