<?php

namespace App\sys\Services\General;

use App\sys\Repository\General\CityRepository;
use App\sys\Services;
use Illuminate\Support\Facades\Validator;

class CityService extends Services
{
    protected CityRepository $cityRepository;

    public function __construct(CityRepository $cityRepository)
    {
        $this->cityRepository = $cityRepository;
    }

    public function getPaginatedCities()
    {
        return $this->cityRepository->getPaginated();
    }

    public function getCityById(int $id)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:cities,id'],
        ];

        $validator = Validator::make(['id' => $id], $rules);

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->cityRepository->findByIdOrFail($id);
    }

    public function createCity($request)
    {
        $rules = [
            'country_id' => ['required', 'integer', 'exists:countries,id'],
            'name' => ['required', 'string', 'max:255'],
            'shortcut' => ['required', 'string', 'max:10', 'unique:cities,shortcut'],
            'active' => ['required', 'in:0,1'],
        ];

        $validator = Validator::make($request, $rules);

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        $city = $this->cityRepository->create($request);
        $this->syncTranslations($city);

        return $city;
    }

    public function updateCity($request)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:cities,id'],
            'country_id' => ['sometimes', 'integer', 'exists:countries,id'],
            'name' => ['sometimes', 'string', 'max:255'],
            'shortcut' => ['sometimes', 'string', 'max:10'],
            'active' => ['sometimes', 'in:0,1'],
        ];

        $validator = Validator::make($request, $rules);

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        $city = $this->cityRepository->findByIdOrFail($request['id']);

        return $this->cityRepository->update($city, $request);
    }

    public function deleteCity(int $id)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:cities,id'],
        ];

        $validator = Validator::make(['id' => $id], $rules);

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        $city = $this->cityRepository->findByIdOrFail($id);

        return $this->cityRepository->delete($city);
    }

    public function del(array $ids)
    {
        return $this->cityRepository->del($ids);
    }

    public function getActive()
    {
        return $this->cityRepository->getActive();
    }

    public function getByIdWithTranslation($id)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:cities,id'],
        ];
        $validator = Validator::make(['id' => $id], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }
        $city = $this->cityRepository->getByIdWithTranslation($id);

        return $this->syncTranslations($city, true);
    }

    public function getCitiesByCountry($countryId)
    {
        $rules = [
            'country_id' => ['required', 'integer', 'exists:countries,id'],
        ];
        $validator = Validator::make(['country_id' => $countryId], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->cityRepository->getCitiesByCountry($countryId);
    }

    public function getAllChanges($city)
    {
        return $this->cityRepository->getAllChanges($city);
    }
}
