<?php

namespace App\sys\Services\General;

use App\sys\Repository\General\ServiceTranslationRepository;
use App\sys\Services;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;

class ServiceTranslationService extends Services
{
    private $serviceTranslation;

    public function __construct()
    {
        $this->serviceTranslation = new ServiceTranslationRepository;
    }

    public function save(array $data, $serviceId)
    {
        $rules = ['services_id' => 'required|integer|exists:services,id'];
        $combinations = [];
        $customErrors = [];

        foreach ($data as $index => $row) {
            $rowId = $row['id'] ?? null;
            $languageId = $row['language_id'] ?? null;
            $rules["$index.id"] = 'sometimes|nullable|exists:services_translation,id';
            $rules["$index.language_id"] = [
                Rule::requiredIf(empty($rowId)),
                'exists:languages,id',
                Rule::unique('services_translation', 'lang_id')
                    ->where(fn ($query) => $query->where('services_id', $serviceId))
                    ->ignore($rowId),
            ];
            $rules["$index.name"] = 'nullable|string|max:255';
            if ($languageId) {
                $key = $serviceId.'-'.$languageId;
                if (in_array($key, $combinations)) {
                    $customErrors["$index.language_id"] = 'Same language duplicated for the same Service.';
                } else {
                    $combinations[] = $key;
                }
            }
        }

        $validator = Validator::make(array_merge($data, ['services_id' => $serviceId]), $rules);

        $validator->after(function ($validator) use ($customErrors) {
            foreach ($customErrors as $field => $message) {
                $validator->errors()->add($field, $message);
            }
        });

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->serviceTranslation->save($data, $serviceId);
    }

    public function get($serviceId)
    {
        return $this->serviceTranslation->get($serviceId);
    }

    public function del($id)
    {
        $rules = [
            'id' => 'required|integer|exists:services_translation,id',
        ];
        $validator = Validator::make(['id' => $id], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->serviceTranslation->del($id);
    }
}
