<?php

namespace App\sys\Services\General;

use App\sys\Repository\General\TaxRateTranslationRepository;
use App\sys\Services;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;

class TaxRateTranslationService extends Services
{
    private $taxRateTranslation;

    public function __construct()
    {
        $this->taxRateTranslation = new TaxRateTranslationRepository;
    }

    public function save(array $data, $id)
    {
        $rules = ['tax_rate_id' => 'required|integer|exists:taxs_rate,id'];
        $combinations = [];
        $customErrors = [];

        foreach ($data as $index => $row) {
            $rowId = $row['id'] ?? null;
            $languageId = $row['language_id'] ?? null;
            $rules["$index.id"] = 'sometimes|nullable|exists:tax_rate_translation,id';
            $rules["$index.language_id"] = [
                Rule::requiredIf(empty($rowId)),
                'exists:languages,id',
                Rule::unique('tax_rate_translation', 'lang_id')
                    ->where(fn ($query) => $query->where('tax_rate_id', $id))
                    ->ignore($rowId),
            ];
            $rules["$index.name"] = 'nullable|string|max:255';
            if ($languageId) {
                $key = $id.'-'.$languageId;
                if (in_array($key, $combinations)) {
                    $customErrors["$index.language_id"] = 'Same language distorted with the same Tax Rate.';
                } else {
                    $combinations[] = $key;
                }
            }
        }

        $validator = Validator::make(array_merge($data, ['tax_rate_id' => $id]), $rules);

        $validator->after(function ($validator) use ($customErrors) {
            foreach ($customErrors as $field => $message) {
                $validator->errors()->add($field, $message);
            }
        });

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->taxRateTranslation->save($data, $id);
    }

    public function get($id)
    {
        return $this->taxRateTranslation->get($id);
    }

    public function del($id)
    {
        $rules = [
            'id' => 'required|integer|exists:tax_rate_translation,id',
        ];
        $validator = Validator::make(['id' => $id], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->taxRateTranslation->del($id);
    }
}
