<?php

namespace App\Http\Controllers\General;

use App\Http\Controllers\Controller;
use App\Http\Resources\CompanyCollection;
use App\Http\Resources\CompanyResource;
use App\Http\Resources\General\Company\CompanyLangResources;
use App\Http\Resources\General\CompanyHistoryResource;
use App\sys\ApiResponse;
use App\sys\Services\General\CompanyService;
use App\sys\Services\General\CompanyTranslationService;
use Illuminate\Http\Request;

class CompanyController extends Controller
{
    use ApiResponse;

    protected CompanyService $companyService;

    protected CompanyTranslationService $companyTranslationService;

    public function __construct(CompanyService $companyService, CompanyTranslationService $companyTranslationService)
    {
        $this->companyService = $companyService;
        $this->companyTranslationService = $companyTranslationService;
    }

    public function index()
    {
        $companies = $this->companyService->getPaginatedCompanies();

        return $this->apiResponse(200, 'success', null, new CompanyCollection($companies));
    }

    public function show(int $id)
    {
        if ($company = $this->companyService->getCompanyById($id)) {
            return $this->apiResponse(200, 'success', null, new CompanyResource($company));
        }
        $errors = $this->companyService->errors();

        return $this->apiResponse(400, 'fail', $errors, null);
    }

    public function store(Request $request)
    {
        if ($response = $this->companyService->createCompany($request->all())) {
            return $this->apiResponse(200, 'success', null, new CompanyResource($response));
        }
        $errors = $this->companyService->errors();

        return $this->apiResponse(400, 'fail', $errors, null);
    }

    public function update(Request $request)
    {

        if ($response = $this->companyService->updateCompany($request->all())) {
            return $this->apiResponse(200, 'success', null, new CompanyResource($response));
        }
        $errors = $this->companyService->errors();

        return $this->apiResponse(400, 'fail', $errors, null);
    }

    public function getCurrencyWithTree()
    {
        $data = $this->companyService->getCurrencyWithtree();

        return $this->apiResponse(200, 'success', null, $data);
    }

    public function getActive()
    {
        $companies = $this->companyService->getActive();

        return $this->apiResponse(200, 'success', null, new CompanyCollection($companies));
    }

    public function destroy(Request $request)
    {
        $ids = $request->input('ids', []);
        if (! is_array($ids) || empty($ids)) {
            return $this->apiResponse(400, 'No IDs provided for deletion', null, null);
        }
        $result = $this->companyService->del($ids);
        if ($result) {
            return $this->apiResponse(200, 'Company  deleted successfully', null, null);
        }
        $errors = $this->companyService->errors();

        return $this->apiResponse(400, 'fail', $errors, null);
    }

    public function SaveTranslation(Request $request, $id)
    {
        if (! $this->companyTranslationService->save($request->all(), $id)) {
            $errors = $this->companyTranslationService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, null);
    }

    public function getTranslation($id)
    {
        $data = $this->companyService->getByidWithTranslation($id);
        $collection = new CompanyLangResources($data);

        return $this->apiResponse(200, 'success', null, $collection);
    }

    public function delLang($id)
    {
        if ($this->companyTranslationService->del($id)) {
            return $this->apiResponse(200, 'success', null, null);
        }
        $errors = $this->companyTranslationService->errors();

        return $this->apiResponse(400, 'fail', $errors, null);
    }

    public function getAllChanges($id)
    {
        $company = $this->companyService->getCompanyById($id);
        if (! $company) {
            $errors = $this->companyService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        $changes = $this->companyService->getAllChanges($company);

        return $this->apiResponse(200, 'success', null, CompanyHistoryResource::collection($changes));
    }
}
