<?php

namespace App\Http\Controllers\General;

use App\Http\Controllers\Controller;
use App\Http\Resources\CountryCollection;
use App\Http\Resources\CountryResource;
use App\Http\Resources\General\CountryHistoryResource;
use App\sys\ApiResponse;
use App\sys\Services\General\CityService;
use App\sys\Services\General\CountryService;
use App\sys\Services\General\CountryTranslationService;
use Illuminate\Http\Request;

class CountryController extends Controller
{
    use ApiResponse;

    protected CountryService $countryService;

    protected CountryTranslationService $countryTranslationService;

    protected CityService $cityService;

    public function __construct(CountryService $countryService, CountryTranslationService $countryTranslationService, CityService $cityService)
    {
        $this->countryService = $countryService;
        $this->countryTranslationService = $countryTranslationService;
        $this->cityService = $cityService;
    }

    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $countries = $this->countryService->getPaginatedCountries();

        return $this->apiResponse(200, 'success', null, new CountryCollection($countries));
    }

    /**
     * Display the specified resource.
     */
    public function show(int $id)
    {
        $country = $this->countryService->getCountryById($id);

        if (! $country) {
            $errors = $this->countryService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, new CountryResource($country));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $response = $this->countryService->createCountry($request->all());

        if (! $response) {
            $errors = $this->countryService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, new CountryResource($response));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request)
    {
        $response = $this->countryService->updateCountry($request->all());

        if (! $response) {
            $errors = $this->countryService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, new CountryResource($response));
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Request $request)
    {
        $ids = $request->input('ids', []);
        if (! is_array($ids) || empty($ids)) {
            return $this->apiResponse(400, 'No IDs provided for deletion', null, null);
        }
        $result = $this->countryService->del($ids);
        if ($result) {
            return $this->apiResponse(200, 'Countries deleted successfully', null, null);
        }
        $errors = $this->countryService->errors();

        return $this->apiResponse(400, 'fail', $errors, null);
    }

    public function getActive()
    {
        $data = $this->countryService->getActive();
        $collect = new \App\Http\Resources\CountryCollection($data);

        return $this->apiResponse(200, 'Active Countries retrieved successfully', null, $collect);
    }

    public function saveTranslation(Request $request, $id)
    {
        if (! $this->countryTranslationService->save($request->all(), $id)) {
            $errors = $this->countryTranslationService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, null);
    }

    public function getTranslation($id)
    {
        if ($data = $this->countryService->getByIdWithTranslation($id)) {
            $collection = new \App\Http\Resources\General\Country\CountryLangResource($data);

            return $this->apiResponse(200, 'success', null, $collection);
        }
        $errors = $this->countryService->errors();

        return $this->apiResponse(400, 'fail', $errors, null);
    }

    public function delLang($id)
    {
        if ($this->countryTranslationService->del($id)) {
            return $this->apiResponse(200, 'success', null, null);
        }

        $errors = $this->countryTranslationService->errors();

        return $this->apiResponse(400, 'fail', $errors, null);
    }

    /**
     * Get cities for a specific country
     */
    public function getCities($countryId)
    {
        $cities = $this->cityService->getCitiesByCountry($countryId);

        if (! $cities) {
            $errors = $this->cityService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, new \App\Http\Resources\CityCollection($cities));
    }

    public function getAllChanges($id)
    {
        $country = $this->countryService->getCountryById($id);
        if (! $country) {
            $errors = $this->countryService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        $changes = $this->countryService->getAllChanges($country);

        return $this->apiResponse(200, 'success', null, CountryHistoryResource::collection($changes));
    }
}
