<?php

namespace App\Http\Controllers\Profile;

use App\Http\Controllers\Controller;
use App\Http\Resources\Profile\ProfileCollection;
use App\Http\Resources\Profile\ProfileColumnHistoryResource;
use App\Http\Resources\Profile\ProfileHistoryResource;
use App\Http\Resources\Profile\ProfileResource;
use App\sys\ApiResponse;
use App\sys\Services\Accounting\UnpostedCollectionsServices;
use App\sys\Services\Profile\ProfileService;
use Illuminate\Http\Request;

class ProfileController extends Controller
{
    use ApiResponse;

    protected ProfileService $profileService;

    public function __construct(ProfileService $profileService)
    {
        $this->profileService = $profileService;

    }

    public function index()
    {
        $items = $this->profileService->getPaginated();

        return $this->apiResponse(200, 'success', null, new ProfileCollection($items));
    }

    public function show($id)
    {
        $item = $this->profileService->getById($id);
        if (! $item) {
            $errors = $this->profileService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }
        $resource = (new ProfileResource($item))->includeChanges();

        return $this->apiResponse(200, 'success', null, $resource);
    }

    public function store(Request $request)
    {
        $response = $this->profileService->create($request->all());
        if (! $response) {
            $errors = $this->profileService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, new ProfileResource($response));
    }

    public function update(Request $request)
    {
        $response = $this->profileService->update($request->all());
        if (! $response) {
            $errors = $this->profileService->errors();

            return $this->apiResponse(400, 'fail', null, $errors);
        }

        return $this->apiResponse(200, 'success', null, new ProfileResource($response));
    }

    public function destroy(Request $request)
    {
        $response = $this->profileService->del($request->ids);
        if (! $response) {
            $errors = $this->profileService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', 'Profile deleted successfully', null);
    }

    public function getActive()
    {
        $items = $this->profileService->getActive();

        return $this->apiResponse(200, 'success', null, new ProfileCollection($items));
    }

    public function getAllChanges($id)
    {
        $allChanges = $this->profileService->getAllChanges($id);
        if (! $allChanges) {
            $errors = $this->profileService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, ProfileHistoryResource::collection($allChanges));
    }

    public function getColumChanges($id, $columnName)
    {
        $columnHistory = $this->profileService->getColumnChanges($id, $columnName);
        if (! $columnHistory) {
            $errors = $this->profileService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, ProfileColumnHistoryResource::collection($columnHistory));
    }

    public function checkReservation($id)
    {
        $data = $this->profileService->checkReservations($id);

        return $this->apiResponse(200, 'success', null, $data);

    }

    public function getProfileShare($code = 0)
    {
        if ($data = $this->profileService->getProfileShare($code)) {
            return $this->apiResponse(200, 'success', null, $data);
        }

        $errors = $this->profileService->errors();

        return $this->apiResponse(400, 'fail', $errors, null);

    }

    public function invoiceProfile($id)
    {
        $accountingService = new UnpostedCollectionsServices;
        $invoice = $accountingService->createInvoiceWithoutPayment((int) $id);

        if (! $invoice) {
            return $this->apiResponse(400, 'fail', $accountingService->errors(), null);
        }

        return $this->apiResponse(200, 'success', null, $invoice);
    }

    public function ProfileNotes($id = 0)
    {
        if ($profile = $this->profileService->notes($id)) {
            return $this->apiResponse(200, 'success', null, $profile);
        }

        $errors = $this->profileService->errors();

        return $this->apiResponse(400, 'fail', $errors, null);
    }

    public function getCurrencyByInvoice($id)
    {
        $currencies = $this->profileService->getCurrencyByInvoice((int) $id);
        if (! $currencies) {
            return $this->apiResponse(400, 'fail', $this->profileService->errors(), null);
        }

        return $this->apiResponse(200, 'success', null, $currencies);
    }

    public function getConstCenter($id)
    {
        $invoiceServicesRepo = new \App\sys\Repository\Invoice\InvoiceServicesRepository;
        $unpostedCollectionsService = new UnpostedCollectionsServices;

        // Get all invoice services for this profile with all relationships
        $allServices = \App\Models\invoice\InvoiceServices::with([
            'service.currentTranslation',
            'supplier.currentTranslation',
            'reservation.accommodation.currentTranslation',
            'taxes.tax.currentTranslation',
            'taxes.currency.currentTranslation',
            'sellingTaxes.tax.currentTranslation',
            'sellingTaxes.currency.currentTranslation',
            'invoice',
            'currency.currentTranslation',
            'sellingCurrency.currentTranslation',
        ])
            ->where('profile_id', $id)
            ->get();

        // Format service data
        $formatService = function ($service) {
            // Get service type and name
            $serviceType = $service->travel_tourism_type;
            $serviceTitle = null;

            if ($serviceType === 'accommodation') {
                // For accommodation, get hotel/cruise type and name
                $accommodationType = null;
                if ($service->reservation && $service->reservation->accommodation) {
                    $accommodation = $service->reservation->accommodation;
                    $accommodationType = $accommodation->type ?? null; // hotel or cruise
                    $serviceTitle = $accommodation->currentTranslation->name
                        ?? $accommodation->name
                        ?? null;
                }
                $serviceType = $accommodationType ?? 'accommodation';
            } else {
                // For other services, get service name
                $serviceTitle = $service->service
                    ? ($service->service->currentTranslation->title ?? $service->service->title ?? null)
                    : null;
            }

            // Get supplier name
            $supplierName = $service->supplier
                ? ($service->supplier->currentTranslation->supplier_name
                    ?? $service->supplier->supplier_name
                    ?? null)
                : null;

            // Format currency object
            $currency = null;
            if ($service->currency) {
                $currency = [
                    'id' => $service->currency->id,
                    'name' => $service->currency->currentTranslation->name
                        ?? $service->currency->name
                        ?? null,
                    'code' => $service->currency->code ?? null,
                    'symbol' => $service->currency->currentTranslation->symbol
                        ?? $service->currency->symbol
                        ?? null,
                    'exchange_rate' => (float) ($service->currency->exchange_rate ?? 0),
                ];
            }

            // Format taxes
            $taxes = $service->taxes->map(function ($tax) {
                $taxRateName = null;
                if ($tax->tax) {
                    $taxRateName = $tax->tax->currentTranslation->title
                        ?? $tax->tax->title
                        ?? null;
                }

                $taxCurrency = null;
                if ($tax->currency) {
                    $taxCurrency = [
                        'id' => $tax->currency->id,
                        'name' => $tax->currency->currentTranslation->name
                            ?? $tax->currency->name
                            ?? null,
                        'code' => $tax->currency->code ?? null,
                        'symbol' => $tax->currency->currentTranslation->symbol
                            ?? $tax->currency->symbol
                            ?? null,
                        'exchange_rate' => (float) ($tax->currency->exchange_rate ?? 0),
                    ];
                }

                return [
                    'id' => $tax->id,
                    'tax_rate_id' => $tax->tax_rate_id,
                    'tax_rate_name' => $taxRateName,
                    'tax_amount' => (float) ($tax->tax_amount ?? 0),
                    'tax_rate_amount' => (float) ($tax->tax_rate_amount ?? 0),
                    'currency' => $taxCurrency,
                ];
            })->values();

            // Base service data
            $serviceData = [
                'id' => $service->id,
                'travel_tourism_type' => $service->travel_tourism_type,
                'service_type' => $serviceType,
                'title' => $serviceTitle,
                'status' => $service->status ?? null,
                'supplier_id' => $service->supplier_id,
                'supplier_name' => $supplierName,
                'paid_to_supplier' => (float) ($service->paid_to_supplier ?? 0),
                'remaining_to_supplier' => (float) ($service->remaining_to_supplier ?? 0),
                'currency' => $currency,
                'currency_rate' => (float) ($service->currency_rate ?? 0),
                'taxes' => $taxes,
                'invoice_id' => $service->invoice_id,
                'serial_num' => $service->invoice ? ($service->invoice->serial_num ?? null) : null,
            ];

            // If is_by_handling == 1, add selling price and selling taxes
            if ($service->is_by_handling == 1) {
                $serviceData['selling_price'] = (float) ($service->selling_price ?? 0);
                $serviceData['selling_rate'] = (float) ($service->selling_rate ?? 0);

                // Get selling currency
                $sellingCurrency = null;
                if ($service->sellingCurrency) {
                    $sellingCurrency = [
                        'id' => $service->sellingCurrency->id,
                        'name' => $service->sellingCurrency->currentTranslation->name
                            ?? $service->sellingCurrency->name
                            ?? null,
                        'code' => $service->sellingCurrency->code ?? null,
                        'symbol' => $service->sellingCurrency->currentTranslation->symbol
                            ?? $service->sellingCurrency->symbol
                            ?? null,
                        'exchange_rate' => (float) ($service->sellingCurrency->exchange_rate ?? 0),
                    ];
                }
                $serviceData['selling_currency'] = $sellingCurrency;

                $sellingTaxes = $service->sellingTaxes->map(function ($tax) {
                    $taxRateName = null;
                    if ($tax->tax) {
                        $taxRateName = $tax->tax->currentTranslation->title
                            ?? $tax->tax->title
                            ?? null;
                    }

                    $taxCurrency = null;
                    if ($tax->currency) {
                        $taxCurrency = [
                            'id' => $tax->currency->id,
                            'name' => $tax->currency->currentTranslation->name
                                ?? $tax->currency->name
                                ?? null,
                            'code' => $tax->currency->code ?? null,
                            'symbol' => $tax->currency->currentTranslation->symbol
                                ?? $tax->currency->symbol
                                ?? null,
                            'exchange_rate' => (float) ($tax->currency->exchange_rate ?? 0),
                        ];
                    }

                    return [
                        'id' => $tax->id,
                        'tax_rate_id' => $tax->tax_rate_id,
                        'tax_rate_name' => $taxRateName,
                        'tax_amount' => (float) ($tax->tax_amount ?? 0),
                        'tax_rate_amount' => (float) ($tax->tax_rate_amount ?? 0),
                        'currency' => $taxCurrency,
                    ];
                })->values();

                $serviceData['selling_taxes'] = $sellingTaxes;
            }

            return $serviceData;
        };

        // Separate services by is_by_handling
        $services = $allServices
            ->where('is_by_handling', 0)
            ->map($formatService)
            ->values();

        $servicesCritical = $allServices
            ->where('is_by_handling', 1)
            ->map($formatService)
            ->values();

        // Get summary
        $summary = $unpostedCollectionsService->summary((int) $id);

        return $this->apiResponse(200, 'success', null, [
            'services' => $services,
            'services_critical' => $servicesCritical,
            'summary' => $summary,
        ]);
    }
}
