<?php

namespace App\sys\Repository\Accounting;

use App\Models\Accounting\Constraint;
use App\Models\Accounting\Transfer;
use App\Models\General\Currency;
use Carbon\Carbon;
use NumberFormatter;

class ConstraintRepository
{
    protected $capture_exchange_types = ['receipt', 'payment'];

    public function getNextDocumentNumber($type)
    {
        $maxNumber = Constraint::where('capture_exchange', $type)
            ->whereNotNull('number_doc')
            ->max('number_doc');

        return ($maxNumber ?? 0) + 1;
    }

    public function create($data, $wallet = null)
    {
        $constraint = new Constraint;
        $constraint->profile_id = $data['profile_id'] ?? null;
        $constraint->year = Carbon::parse($data['date'])->year;
        $constraint->type_of_movement_id = $data['type_of_movement_id'] ?? null;
        $constraint->number_movement = $data['number_movement'] ?? null;
        $constraint->date = $data['date'] ?? null;
        $constraint->name = $data['name'] ?? null;
        $constraint->number_doc = $this->getNextDocumentNumber($data['capture_exchange']);
        $constraint->description = $data['description'] ?? null;
        $constraint->creation_mode = $data['creation_mode'] ?? 'manual';
        $constraint->active = $data['active'] ?? null;
        $constraint->type_of_movement = $data['type_of_movement'] ?? null;
        $constraint->capture_exchange = $data['capture_exchange'];
        $constraint->type_optional = $data['type_optional'] ?? null;
        $constraint->number_chek = $data['number_chek'] ?? null;
        $constraint->chek_date = $data['chek_date'] ?? null;
        $constraint->username = $data['username'] ?? null;
        $constraint->company_id = $data['company_id'] ?? null;
        $constraint->customer_id = $data['customer_id'] ?? null;
        $constraint->supplier_id = $data['supplier_id'] ?? null;
        $constraint->user_id_emp = $data['user_id_emp'] ?? null;
        $constraint->invoice_id = $data['invoice_id'] ?? null;
        $constraint->duration = $data['duration'] ?? null;
        $constraint->due_date = $data['due_date'] ?? null;
        $constraint->worthy_days = $data['worthy_days'] ?? null;
        $constraint->payType_id = $data['payType_id'] ?? null;
        $constraint->employee_tax = $data['employee_tax'] ?? null;
        $constraint->pay_tax = $data['pay_tax'] ?? null;
        $constraint->currency_id = $data['currency_id'];
        $constraint->currency_transfer_rate = Currency::find($data['currency_id'])->exchange_rate ?? 1;
        $constraint->total_creditor = $data['total_creditor'];
        $constraint->total_debit = $data['total_debit'];
        $constraint->difference = $data['difference'] ?? 0;
        $constraint->sales_id = $data['sales_id'] ?? 0;
        $constraint->invoice_services_id = $data['invoice_services_id'] ?? null;
        $constraint->user_id = $data['user_id'] ?? Auth()->user()->id; // change it later to the authenticated user but for now keep it till the auth is done.
        $constraint->save();
        $constraintCurrencyDebit = 0;
        $constraintCurrencyCreditor = 0;
        foreach ($data['transfers'] as $transferData) {
            $transfer = new Transfer;

            $transfer->constraint_id = $constraint->id;
            $transfer->supplier_id = $constraint->supplier_id;
            $transfer->name = $transferData['name'];
            $transfer->pay_type_id = $constraint->payType_id;
            $transfer->invoice_services_id = $constraint->invoice_services_id;
            $transfer->tree_accounting_id = $transferData['tree_accounting_id'];
            $transfer->capture_exchange = $constraint->capture_exchange;
            $transfer->type_optional = $constraint->type_optional;
            $transfer->customer_id = $constraint->customer_id;
            $transfer->user_id = $constraint->user_id ?? null;
            $transfer->sales_id = $constraint->sales_id;
            $transfer->cost_center_id = in_array($constraint->capture_exchange, $this->capture_exchange_types)
                ? null
                : $transferData['cost_center_id'];
            $transfer->date = $constraint->date;

            if ($wallet == null) {
                // تحديد العملة الافتراضية وعملة القيد

                $constraintCurrencyId = $constraint->currency_id; // قد تكون null
                $constraintRate = $constraint->currency_transfer_rate ?? 1;

                // القيم المرسلة
                $rawCreditor = (float) ($transferData['creditor'] ?? 0);
                $rawDebit = (float) ($transferData['debit'] ?? 0);

                // منطق التحويل:
                // 1) إذا لم تُرسل currency_id (القيد بدون عملة) فاعتبر المبالغ بالعملة الافتراضية
                //    واحفظها كما هي، لكن إن كانت عملة الحساب مختلفة يمكن الاحتفاظ بالأصل في نفس الحقول
                // 2) إذا أُرسلت عملة للقيد وهي ليست الافتراضية: نحول القيم إلى الافتراضية بالضرب في rate
                //    ونخزن القيم المرسلة كما هي في حقول العملة الخاصة بالقيد عبر currency_id و currency_transfer_rate

                if ($data['currency_id'] != null) {
                    // القيم المرسلة بعملة غير افتراضية -> حول للأرقام الأساسية بالافتراضية بضرب rate
                    $transfer->creditor = $rawCreditor * $constraintRate;
                    $transfer->debit = $rawDebit * $constraintRate;
                    $transfer->currency_creditor = $rawCreditor;
                    $transfer->currency_debit = $rawDebit;
                    $transfer->currency_id = $constraintCurrencyId;
                    $transfer->currency_transfer_rate = $constraintRate;
                } else {
                    // القيم بالعملة الافتراضية
                    $tree = \App\Models\Accounting\TreeAccounting::find($transferData['tree_accounting_id']);
                    if ($tree) {
                        $treeCurrencyId = $tree->currency_id;
                        $transRate = Currency::find($tree->currency_id)->exchange_rate ?? $constraintRate;
                    }
                    $transfer->creditor = $rawCreditor;
                    $transfer->debit = $rawDebit;
                    $transfer->currency_creditor = $rawCreditor / $transRate;
                    $transfer->currency_debit = $rawDebit / $transRate;
                    $transfer->currency_id = $treeCurrencyId;
                    $transfer->currency_transfer_rate = $transRate;
                }

                // خزّن عملة القيد ومعدل التحويل المرجعيين على التحويل

            } else {
                $transfer->creditor = $transferData['creditor'];
                $transfer->debit = $transferData['debit'];
                $transfer->currency_creditor = $transferData['currency_creditor'];
                $transfer->currency_debit = $transferData['currency_debit'];
                $transfer->currency_id = $transferData['currency_id'] ?? $data['currency_id'];
                $transfer->currency_transfer_rate = $transferData['currency_transfer_rate'] ?? $data['currency_transfer_rate'];
            }
            $transfer->save();
            $constraintCurrencyDebit += $transfer->currency_debit;
            $constraintCurrencyCreditor += $transfer->currency_creditor;
        }

        if ($constraint->currency_id != null) {
            $constraint->currency_creditor = $constraintCurrencyCreditor;
            $constraint->currency_debit = $constraintCurrencyDebit;
            $constraint->save();
        }

        return $constraint->load(['user', 'transfers.currency.currentTranslation', 'transfers.treeAccounting', 'transfers.costCenter', 'profile', 'invoice']);
    }

    public function update(Constraint $constraint, array $data)
    {
        if (isset($data['profile_id'])) {
            $constraint->profile_id = $data['profile_id'];
        }
        $constraint->type_of_movement_id = $data['type_of_movement_id'] ?? $constraint->type_of_movement_id;
        $constraint->number_movement = $data['number_movement'] ?? $constraint->number_movement;
        if (isset($data['date'])) {
            $constraint->date = $data['date'];
            $constraint->year = Carbon::parse($data['date'])->year;
        }
        $constraint->name = $data['name'] ?? $constraint->name;
        $constraint->description = $data['description'] ?? $constraint->description;
        $constraint->active = $data['active'] ?? $constraint->active;
        $constraint->type_of_movement = $data['type_of_movement'] ?? $constraint->type_of_movement;
        $constraint->number_chek = $data['number_chek'] ?? $constraint->number_chek;
        $constraint->chek_date = $data['chek_date'] ?? $constraint->chek_date;
        $constraint->username = $data['username'] ?? $constraint->username;
        $constraint->company_id = $data['company_id'] ?? $constraint->company_id;
        $constraint->customer_id = $data['customer_id'] ?? $constraint->customer_id;
        $constraint->supplier_id = $data['supplier_id'] ?? $constraint->supplier_id;
        $constraint->user_id_emp = $data['user_id_emp'] ?? $constraint->user_id_emp;
        $constraint->invoice_id = $data['invoice_id'] ?? $constraint->invoice_id;
        $constraint->duration = $data['duration'] ?? $constraint->duration;
        $constraint->due_date = $data['due_date'] ?? $constraint->due_date;
        $constraint->worthy_days = $data['worthy_days'] ?? $constraint->worthy_days;
        $constraint->payType_id = $data['payType_id'] ?? $constraint->payType_id;
        $constraint->employee_tax = $data['employee_tax'] ?? $constraint->employee_tax;
        $constraint->pay_tax = $data['pay_tax'] ?? $constraint->pay_tax;
        /*
        if (isset($data['currency_id'])) {
            $constraint->currency_id = $data['currency_id'];
            $constraint->currency_transfer_rate = Currency::find($data['currency_id'])->exchange_rate ?? 1;
        }
        */
        $constraint->total_creditor = $data['total_creditor'];
        $constraint->total_debit = $data['total_debit'];
        $constraint->difference = $data['difference'] ?? $constraint->difference;
        $constraint->save();
        $constraintCurrencyDebit = 0;
        $constraintCurrencyCreditor = 0;
        Transfer::where('constraint_id', $constraint->id)->delete();
        foreach ($data['transfers'] as $transferData) {
            $transfer = new Transfer;
            $transfer->constraint_id = $constraint->id;
            $transfer->name = $transferData['name'];
            $transfer->tree_accounting_id = $transferData['tree_accounting_id'];
            $transfer->cost_center_id = in_array($constraint->capture_exchange, $this->capture_exchange_types)
                ? null
                : $transferData['cost_center_id'];
            $transfer->date = $constraint->date;

            $constraintCurrencyId = $constraint->currency_id; // may be null
            $constraintRate = $transferData['currency_transfer_rate'] ?? 1;

            $rawCreditor = (float) ($transferData['creditor'] ?? 0);
            $rawDebit = (float) ($transferData['debit'] ?? 0);

            if ($data['currency_id'] != null) {
                $transfer->creditor = $rawCreditor * $constraintRate;
                $transfer->debit = $rawDebit * $constraintRate;
                $transfer->currency_creditor = $rawCreditor;
                $transfer->currency_debit = $rawDebit;
                $transfer->currency_id = $constraintCurrencyId;
                $transfer->currency_transfer_rate = $constraintRate;
            } else {
                $transfer->creditor = $rawCreditor;
                $transfer->debit = $rawDebit;
                $transfer->currency_creditor = $rawCreditor / $constraintRate;
                $transfer->currency_debit = $rawDebit / $constraintRate;
                $tree = \App\Models\Accounting\TreeAccounting::find($transferData['tree_accounting_id']);
                $transfer->currency_transfer_rate = $constraintRate;
                $transfer->currency_id = $tree->currency_id;
            }

            $transfer->save();
            $constraintCurrencyDebit += $transfer->currency_debit;
            $constraintCurrencyCreditor += $transfer->currency_creditor;
        }

        if ($constraint->currency_id != null) {
            $constraint->currency_creditor = $constraintCurrencyCreditor;
            $constraint->currency_debit = $constraintCurrencyDebit;
            $constraint->save();
        }

        return $constraint->load(['user', 'transfers.currency.currentTranslation', 'transfers.treeAccounting', 'transfers.costCenter', 'profile', 'invoice']);
    }

    public function getConstriantByid($constraint_id)
    {

        $entry = Constraint::with([
            'transfers.treeAccounting',
            'transfers.currency.currentTranslation',
            'transfers.costCenter',
            'attachments',
            'currency.currentTranslation',
            'profile',
            'user',
            'invoice',
        ])->findOrFail($constraint_id);

        // إجماليات الدائن والمدين
        $totalDebit = $entry->transfers->sum('debit');
        $totalCredit = $entry->transfers->sum('creditor');
        $balanced = abs($totalDebit - $totalCredit) < 0.01;

        $data = [
            // Header fields
            'id' => $entry->id,
            'status' => $entry->active ? 'مرحل' : 'غير مرحل',
            'entry_date' => $entry->date,
            'entry_year' => $entry->year,
            'document_no' => $entry->number_doc,
            'exchange_type' => $entry->capture_exchange,
            'description_ar' => $entry->description,
            'description_en' => $entry->description_en ?? null,
            'created_at' => $entry->created_at,
            'updated_at' => $entry->updated_at,

            // Lines (transfers)
            'lines' => $entry->transfers->map(function ($line) {
                return [
                    'account_number' => $line->treeAccounting?->serial_number ?? '----',
                    'line_description_ar' => $line->debit_text ?? $line->creditor_text ?? '----',
                    'line_description_en' => null,
                    'debit' => (float) $line->debit,
                    'credit' => (float) $line->creditor,
                    'cost_center_number' => $line->costCenter?->number ?? '----',
                ];
            }),

            // Totals
            'totals' => [
                'total_debit' => (float) $totalDebit,
                'total_credit' => (float) $totalCredit,
                'balanced' => $balanced,
            ],

            // Attachments
            'attachments' => $entry->attachments->map(function ($file) {
                return [
                    'id' => $file->id,
                    'type' => $file->attachable_type,
                    'created_at' => $file->created_at,
                    'url' => asset('storage/'.$file->path),
                ];
            }),

        ];

        return $data;

    }

    public function print($constraint_id)
    {
        $fmt = new NumberFormatter('en', NumberFormatter::SPELLOUT);
        $constraint = Constraint::with(['user', 'transfers.treeAccounting.translations', 'transfers.currency.currentTranslation', 'company', 'user',  'profile', 'transfers.costCenter', 'invoice'])->find($constraint_id);
        // تحديد نوع السند
        $voucherType = match ($constraint->capture_exchange) {
            'receipt' => 'Cash Receipt Voucher',
            'payment' => 'Cash Payment Voucher',
            'exchange' => 'Cash Exchange Voucher',
            default => 'Cash Voucher',
        };
        $totalDebit = $constraint->transfers->sum('debit');
        $totalCredit = $constraint->transfers->sum('creditor');
        $balanced = $totalDebit == $totalCredit;

        $creditAccount = $constraint->transfers
            ->where('creditor', '!=', 0)
            ->where('debit', 0)
            ->first();

        // ⚙️ تجهيز بيانات الحساب الدائن
        $creditAccountInfo = null;
        if ($creditAccount) {
            $creditAccountInfo = [
                'account_title' => $creditAccount->treeAccounting->title ?? '---',
                'credit_amount' => $creditAccount->creditor,
                'account_id' => $creditAccount->treeAccounting->id ?? null,
                'currency_symbol' => $constraint->currency_id ? Currency::find($constraint->currency_id)->symbol : '',
                'amount_in_words' => ucfirst($fmt->format($creditAccount->creditor)).' Egyptian pounds only',
            ];
        }

        // إضافة كل القيم لنتيجة واحدة
        $constraint['voucher_type'] = $voucherType;
        $constraint['total_debit'] = $totalDebit;
        $constraint['total_credit'] = $totalCredit;
        $constraint['balanced'] = $balanced;
        $constraint['credit_account'] = $creditAccountInfo;

        return [
            'success' => true,
            'data' => $constraint,
        ];
    }

    public function printDailyConstraint($constraint_id)
    {
        $constraint = Constraint::with([
            'transfers.treeAccounting.translations',
            'transfers.costCenter',
            'company',
            'profile',
            'user',
            'invoice',
        ])->find($constraint_id);

        if (! $constraint) {
            return ['success' => false, 'error' => 'Constraint not found'];
        }
        // 1. Header Data
        $header = [
            'constraint_number' => $constraint->number_doc,
            'date' => $constraint->date,
            'ref_number' => $constraint->number_movement ?? '---',
            'description' => $constraint->description, // Arabic description
            'company_name' => $constraint->company->name_company ?? '---',
            'company_vat' => $constraint->company->company_vat ?? '---',
            'type' => $constraint->active ? 'Posted' : 'Unposted',
        ];

        // 2. Transactions Data
        $transactions = $constraint->transfers->map(function ($transfer) {
            return [
                'account_number' => $transfer->treeAccounting->serial_number ?? '---',
                'account_name' => $transfer->treeAccounting->title ?? '---',
                'cost_center_name' => $transfer->costCenter->name ?? '---',
                'description' => $transfer->description ?? '---',
                'debit' => $transfer->debit,
                'credit' => $transfer->creditor,
            ];
        });

        // 3. Totals Data
        $totalDebit = $constraint->transfers->sum('debit');
        $totalCredit = $constraint->transfers->sum('creditor');
        $difference = $totalDebit - $totalCredit;

        $totals = [
            'total_debit' => $totalDebit,
            'total_credit' => $totalCredit,
            'difference' => $difference,
        ];

        return [
            'success' => true,
            'data' => [
                'header' => $header,
                'transactions' => $transactions,
                'totals' => $totals,
            ],
        ];
    }

    public function findById($id)
    {
        return Constraint::with(['user', 'currency.currentTranslation', 'transfers.currency.currentTranslation', 'transfers.treeAccounting', 'attachments', 'profile', 'transfers.costCenter', 'invoice'])->find($id);
    }

    public function del(array $ids)
    {
        Transfer::whereIn('constraint_id', $ids)->delete();

        return Constraint::whereIn('id', $ids)->delete();
    }

    public function activateById(int $id)
    {
        $constraint = Constraint::find($id);
        $constraint->active = 1;
        $constraint->save();

        return $constraint;
    }

    private $columns = [
        'id' => 'id',
        'name' => 'name',
        'date' => 'date',
        'active' => 'active',
        'total_creditor' => 'total_creditor',
        'total_debit' => 'total_debit',
        'type_of_movement' => 'type_of_movement',
        'created_at' => 'created_at',
        'updated_at' => 'updated_at',
    ];

    public function getPaginated()
    {
        $column = request('sort_by', null);
        $order = request('sort_order') ?: 'desc';
        $name = request('name', null);
        $dateFrom = request('date_from', null);
        $dateTo = request('date_to', default: null);
        $exchangeType = request('capture_exchange', null);
        $limit = request('limit', 15);
        $creationMode = request('creation_mode', null);
        $profileId = request('profile_id', null);
        $query = Constraint::query()->with('currency.currentTranslation', 'invoice')
            ->when($name, function ($q, $name) {
                $q->where('name', 'LIKE', "%$name%");
            })
            ->when($dateFrom, function ($q, $dateFrom) {
                $q->whereDate('date', '>=', $dateFrom);
            })
            ->when($dateTo, function ($q, $dateTo) {
                $q->whereDate('date', '<=', $dateTo);
            })
            ->when($exchangeType, function ($q, $exchangeType) {
                $q->where('capture_exchange', $exchangeType);
            })->when($creationMode, function ($q) use ($creationMode) {
                $q->where('creation_mode', $creationMode);
            })->when($profileId, fn ($q) => $q->where('profile_id', $profileId));

        if ($column && array_key_exists($column, $this->columns)) {
            $query->orderBy($this->columns[$column], $order);
        }

        return $query->orderByDesc('updated_at')->paginate($limit);
    }
}
