<?php

namespace App\sys\Services\Invoice\Types;

use App\Models\General\Currency;
use App\Models\General\Service as GeneralService;
use App\Models\General\TaxRate;
use App\Models\invoice\InvoiceServicesSellingTax;
use App\Models\Profile\DailyPrograms;
use App\Models\Profile\Profile;
use App\sys\Enums\ServicesType;
use Illuminate\Support\Facades\Gate;

class DailyTransportationHandler implements InvoiceTypeHandlerInterface
{
    public function getCreateRules(array $request): array
    {
        $basic = $this->authorizeBasic($request, 'add');
        $financial = $this->authorizeFinancial($request, 'add');
        $selling = $this->authorizeSelling($request, 'add');

        return [
            'profile_id' => ['required', 'integer', 'exists:pr_profile,id'],
            'daily_program_id' => ['required', 'integer', 'exists:pr_daily_programs,id'],
            'supplier_id' => [$financial, 'integer', 'exists:su_supplier,id'],
            'city_id' => [$basic, 'integer', 'exists:cities,id'],
            'service_id' => [$basic, 'integer', 'exists:services,id'],
            'people_count' => [$basic, 'integer', 'min:0'],
            'daily_cost' => [$financial, 'numeric', 'min:0'],
            'extra_cost' => ['sometimes', 'nullable', 'numeric', 'min:0'],

            'currency_id' => [$financial, 'integer', 'exists:currencies,id'],
            'operation_data_id' => ['required', 'integer', 'exists:ge_operation_data,id'],
            'executive_id' => ['nullable', 'integer', 'exists:users,id'],
            'notes' => ['nullable', 'string', 'max:1000'],
            'is_by_handling' => ['nullable', 'boolean'],

            // optional taxes array similar to tour_guide
            'tax_rate_id' => ['sometimes', 'nullable', 'array'],
            'tax_rate_id.*' => ['integer', 'exists:taxs_rate,id'],
            // selling fields
            'selling_currency_id' => ['sometimes', $selling, 'integer', 'exists:currencies,id'],
            'selling_tax_rate_id' => ['sometimes', 'nullable', 'array'],
            'selling_tax_rate_id.*' => ['integer', 'exists:taxs_rate,id'],
            'selling_price' => ['sometimes', $selling, 'numeric', 'min:0'],
            // 'selling_total_tax' => ['prohibited'],
        ];
    }

    public function getUpdateRules(array $request): array
    {
        $basic = $this->authorizeBasic($request, 'edit');
        $financial = $this->authorizeFinancial($request, 'edit');
        $selling = $this->authorizeSelling($request, 'edit');

        return [
            'profile_id' => ['sometimes', 'required', 'integer', 'exists:pr_profile,id'],
            'daily_program_id' => ['sometimes', 'required', 'integer', 'exists:pr_daily_programs,id'],
            'supplier_id' => ['sometimes', $financial, 'integer', 'exists:su_supplier,id'],
            'city_id' => ['sometimes', $basic, 'integer', 'exists:cities,id'],
            'service_id' => ['sometimes', $basic, 'integer', 'exists:services,id'],
            'people_count' => ['sometimes', $basic, 'integer', 'min:0'],
            'daily_cost' => ['sometimes', $financial, 'numeric', 'min:0'],
            'extra_cost' => ['sometimes', 'nullable', 'numeric', 'min:0'],
            'currency_id' => ['sometimes', $financial, 'integer', 'exists:currencies,id'],
            'operation_data_id' => ['sometimes', 'required', 'integer', 'exists:ge_operation_data,id'],
            'executive_id' => ['sometimes', 'nullable', 'integer', 'exists:users,id'],
            'notes' => ['sometimes', 'nullable', 'string', 'max:1000'],
            'tax_rate_id' => ['sometimes', 'nullable', 'array'],
            'tax_rate_id.*' => ['integer', 'exists:taxs_rate,id'],
            'is_by_handling' => ['nullable', 'boolean'],
            // selling fields
            'selling_currency_id' => ['sometimes', $selling, 'integer', 'exists:currencies,id'],
            'selling_tax_rate_id' => ['sometimes', 'nullable', 'array'],
            'selling_tax_rate_id.*' => ['integer', 'exists:taxs_rate,id'],
            'selling_price' => ['sometimes', $selling, 'numeric', 'min:0'],
            // 'selling_total_tax' => ['prohibited'],

        ];
    }

    public function prepareForCreate(array $request): array|false
    {
        $this->authorizeBasic($request, 'add');
        $this->authorizeFinancial($request, 'add');
        $canFinancial = Gate::allows('ADD_DAILY_TRANSPORTATION_FINANCIAL_INFO');

        $service = GeneralService::find($request['service_id']);
        if (! $service || $service->type !== ServicesType::DAILY_TRANSPORTATION->value) {
            return ['errors' => ['service_id' => ['service type must be daily_transportation']]];
        }
        $request['is_by_handling'] = (bool) ($request['is_by_handling'] ?? false);

        // Set next_pay_date from daily_program day_date (before financial check)
        if (isset($request['daily_program_id'])) {
            $dailyProgram = DailyPrograms::find($request['daily_program_id']);
            if ($dailyProgram && $dailyProgram->day_date) {
                $request['next_pay_date'] = $dailyProgram->day_date;
            }
        }

        if (! $canFinancial) {
            $request['currency_rate'] = 1;
            $request['total_tax'] = 0;
            $request['grand_total'] = 0;

            return $request;
        }

        // Set currency rate
        $currency = isset($request['currency_id']) ? Currency::find($request['currency_id']) : null;
        $request['currency_rate'] = $currency?->exchange_rate ?? 1;

        $peopleCount = (int) $request['people_count'];
        $dailyCost = (float) ($request['daily_cost'] ?? 0);
        $extraCost = (float) ($request['extra_cost'] ?? 0);
        // Taxes: sum of provided tax rates percentage on subtotal (people * daily_cost)
        $subtotal = $dailyCost;
        $totalTax = 0.0;
        $taxRateIds = $request['tax_rate_id'] ?? [];
        if (is_array($taxRateIds) && ! empty($taxRateIds)) {
            $rates = TaxRate::whereIn('id', $taxRateIds)->pluck('percentage');
            $sumPercent = (float) $rates->sum();
            $totalTax = ($subtotal * $sumPercent) / 100.0;
        }
        $request['total_tax'] = $totalTax;
        $request['grand_total'] = $subtotal + $extraCost + $totalTax;

        // Handle selling (optional)
        $canSelling = Gate::allows('ADD_DAILY_TRANSPORTATION_SELLING_PRICE');
        $profile = isset($request['profile_id']) ? Profile::find($request['profile_id']) : null;
        $eligibleForSelling = $request['is_by_handling'] === true
            || ($profile && (bool) ($profile->has_invoice ?? false));

        if (! $canSelling || ! $eligibleForSelling) {
            $request['selling_total_tax'] = 0;
        } else {
            $sellingCurrencyId = (int) ($request['selling_currency_id'] ?? $request['currency_id'] ?? 0);
            $sellingPrice = (float) ($request['selling_price'] ?? 0);
            $sellingTaxTotal = 0.0;
            $sellingTaxRateIds = $request['selling_tax_rate_id'] ?? [];
            if (is_array($sellingTaxRateIds) && ! empty($sellingTaxRateIds) && $sellingPrice > 0) {
                $rates = TaxRate::whereIn('id', $sellingTaxRateIds)->pluck('percentage');
                $sumPercent = (float) $rates->sum();
                $sellingTaxTotal = ($sellingPrice * $sumPercent) / 100.0;
            }
            // Get selling currency rate
            $sellingCurrency = $sellingCurrencyId ? Currency::find($sellingCurrencyId) : null;
            $request['selling_rate'] = $sellingCurrency?->exchange_rate ?? 1;
            $request['selling_total_tax'] = $sellingTaxTotal;
            $request['selling_currency_id'] = $sellingCurrencyId ?: null;
            $request['selling_price'] = $sellingPrice;
        }

        return $request;
    }

    public function prepareForUpdate(array $request, object $existing): array|false
    {
        $this->authorizeBasic($request, 'edit');
        $this->authorizeFinancial($request, 'edit');
        $canFinancial = Gate::allows('EDIT_DAILY_TRANSPORTATION_FINANCIAL_INFO');

        if (array_key_exists('service_id', $request)) {
            $service = GeneralService::find($request['service_id']);
            if (! $service || $service->type !== ServicesType::DAILY_TRANSPORTATION->value) {
                return ['errors' => ['service_id' => ['service type must be daily_transportation']]];
            }
        }
        $request['is_by_handling'] = (bool) ($request['is_by_handling'] ?? false);
        if (! $canFinancial) {
            return $request;
        }

        $currencyId = (int) ($request['currency_id'] ?? $existing->currency_id);
        if ($currencyId) {
            $currency = Currency::find($currencyId);
            $request['currency_rate'] = $currency?->exchange_rate ?? ($existing->currency_rate ?? 1);
        }

        $peopleCount = (int) ($request['people_count'] ?? $existing->people_count ?? 0);
        $dailyCost = (float) ($request['daily_cost'] ?? $existing->daily_cost ?? 0);
        $extraCost = (float) ($request['extra_cost'] ?? $existing->extra_cost ?? 0);
        $subtotal = $dailyCost;
        $totalTax = 0.0;
        $taxRateIds = $request['tax_rate_id'] ?? null; // if provided, recalc; else keep existing
        if (is_array($taxRateIds)) {
            $rates = TaxRate::whereIn('id', $taxRateIds)->pluck('percentage');
            $sumPercent = (float) $rates->sum();
            $totalTax = ($subtotal * $sumPercent) / 100.0;
        } else {
            $totalTax = (float) ($existing->total_tax ?? 0);
        }
        $request['total_tax'] = $totalTax;
        $request['grand_total'] = $subtotal + $extraCost + $totalTax;

        // Handle selling (optional)
        $canSelling = Gate::allows('EDIT_DAILY_TRANSPORTATION_SELLING_PRICE');
        $profileId = (int) ($request['profile_id'] ?? $existing->profile_id ?? 0);
        $profile = $profileId ? Profile::find($profileId) : null;
        $eligibleForSelling = $request['is_by_handling'] === true
            || ($profile && (bool) ($profile->has_invoice ?? false));

        if ($canSelling && $eligibleForSelling) {
            $sellingCurrencyId = (int) ($request['selling_currency_id'] ?? $existing->selling_currency_id ?? $existing->currency_id ?? 0);
            $sellingPrice = (float) ($request['selling_price'] ?? $existing->selling_price ?? 0);
            $sellingTaxRateIds = $request['selling_tax_rate_id'] ?? null;
            if (is_array($sellingTaxRateIds) && $sellingPrice > 0) {
                $rates = TaxRate::whereIn('id', $sellingTaxRateIds)->pluck('percentage');
                $sumPercent = (float) $rates->sum();
                $sellingTotalTax = ($sellingPrice * $sumPercent) / 100.0;
            } else {
                $sellingTotalTax = (float) ($existing->selling_total_tax ?? 0);
            }
            // Get selling currency rate
            $sellingCurrency = $sellingCurrencyId ? Currency::find($sellingCurrencyId) : null;
            $request['selling_rate'] = $sellingCurrency?->exchange_rate ?? ($existing->selling_rate ?? 1);
            $request['selling_currency_id'] = $sellingCurrencyId ?: null;
            $request['selling_price'] = $sellingPrice;
            $request['selling_total_tax'] = $sellingTotalTax;
        }

        return $request;
    }

    public function afterCreate(object $invoiceService, array $request): void
    {
        $this->saveTaxes($invoiceService, $request);
        $this->saveSellingTaxes($invoiceService, $request);
    }

    public function afterUpdate(object $invoiceService, array $request): void
    {
        if (array_key_exists('tax_rate_id', $request)) {
            $this->saveTaxes($invoiceService, $request, true);
        }

        if (array_key_exists('selling_tax_rate_id', $request)) {
            $this->saveSellingTaxes($invoiceService, $request, true);
        }
    }

    private function saveTaxes(object $invoiceService, array $request, bool $replace = false): void
    {
        $taxRateIds = $request['tax_rate_id'] ?? [];
        if (! is_array($taxRateIds)) {
            return;
        }

        if ($replace) {
            \App\Models\invoice\InvoiceServicesTax::where('invoice_services_id', $invoiceService->id)->delete();
        }

        $people = (int) ($request['people_count'] ?? $invoiceService->people_count ?? 0);
        $dailyCost = (float) ($request['daily_cost'] ?? $invoiceService->daily_cost ?? 0);
        $subtotal = $people * $dailyCost;

        $rates = TaxRate::whereIn('id', $taxRateIds)->get(['id', 'percentage']);
        foreach ($rates as $rate) {
            $amount = ($subtotal * (float) $rate->percentage) / 100.0;
            \App\Models\invoice\InvoiceServicesTax::create([
                'invoice_services_id' => $invoiceService->id,
                'tax_rate_id' => $rate->id,
                'percentage' => (float) $rate->percentage,
                'tax_amount' => $amount,
            ]);
        }
    }

    private function authorizeBasic(array $request, string $mode): string
    {
        $ability = $mode === 'edit' ? 'VIEW_DAILY_TRANSPORTATION_BASIC_INFO' : 'ADD_DAILY_TRANSPORTATION_BASIC_INFO';

        return Gate::allows($ability) ? 'required' : 'nullable';
    }

    private function authorizeFinancial(array $request, string $mode): string
    {
        $ability = $mode === 'edit' ? 'EDIT_DAILY_TRANSPORTATION_FINANCIAL_INFO' : 'ADD_DAILY_TRANSPORTATION_FINANCIAL_INFO';

        return Gate::allows($ability) ? 'required' : 'nullable';
    }

    private function authorizeSelling(array $request, string $mode): string
    {
        $ability = $mode === 'edit' ? 'EDIT_DAILY_TRANSPORTATION_SELLING_PRICE' : 'ADD_DAILY_TRANSPORTATION_SELLING_PRICE';

        if (! Gate::allows($ability)) {
            return 'nullable';
        }

        // في create: required فقط لو profile->has_invoice == 1
        if ($mode === 'add') {
            $profileId = (int) ($request['profile_id'] ?? 0);
            if ($profileId > 0) {
                $profile = Profile::find($profileId);
                if ($profile && (bool) ($profile->has_invoice ?? false)) {
                    return 'required';
                }
            }

            return 'nullable';
        }

        // في update: required فقط لو is_by_handling == 1
        if ($mode === 'edit') {
            $isByHandling = (bool) ($request['is_by_handling'] ?? false);

            return $isByHandling ? 'required' : 'nullable';
        }

        return 'nullable';
    }

    private function saveSellingTaxes(object $invoiceService, array $request, bool $replace = false): void
    {
        $taxRateIds = $request['selling_tax_rate_id'] ?? [];
        if (! is_array($taxRateIds) || empty($taxRateIds)) {
            if ($replace) {
                InvoiceServicesSellingTax::where('invoice_services_id', $invoiceService->id)->delete();
            }

            return;
        }

        if ($replace) {
            InvoiceServicesSellingTax::where('invoice_services_id', $invoiceService->id)->delete();
        }

        $sellingPrice = (float) ($request['selling_price'] ?? $invoiceService->selling_price ?? 0);
        if ($sellingPrice <= 0) {
            return;
        }

        $profileId = (int) ($invoiceService->profile_id ?? null);
        $currencyId = (int) ($request['selling_currency_id'] ?? $invoiceService->selling_currency_id ?? $invoiceService->currency_id ?? null);

        $rates = TaxRate::whereIn('id', $taxRateIds)->get(['id', 'percentage']);
        foreach ($rates as $rate) {
            $amount = ($sellingPrice * (float) $rate->percentage) / 100.0;
            InvoiceServicesSellingTax::create([
                'invoice_services_id' => $invoiceService->id,
                'tax_rate_id' => $rate->id,
                'tax_amount' => $amount,
                'tax_rate_amount' => (float) $rate->percentage,
                'currency_id' => $currencyId ?: null,
                'profile_id' => $profileId ?: null,
            ]);
        }
    }
}
