<?php

namespace App\sys\Services\Invoice\Types;

use App\Models\General\Currency;
use App\Models\General\Service as GeneralService;
use App\Models\General\TaxRate;
use App\Models\invoice\InvoiceServicesSellingTax;
use App\Models\Profile\DailyPrograms;
use App\Models\Profile\Profile;
use App\sys\Enums\ServicesType;
use Illuminate\Support\Facades\Gate;

class TourRepsHandler implements InvoiceTypeHandlerInterface
{
    public function getCreateRules(array $request): array
    {
        $basic = $this->authorizeBasic($request, 'add');
        $financial = $this->authorizeFinancial($request, 'add');
        $selling = $this->authorizeSelling($request, 'add');

        $rules = [
            'service_id' => [$basic, 'integer', 'exists:services,id'],
            'service_provider' => [$financial, 'in:supplier_id,guide_id'],
        ];

        // Conditional validation based on service_provider
        if ($financial === 'required') {
            // If financial is required, then one of supplier_id or guide_id must be provided based on service_provider
            // supplier_id is required when service_provider = supplier_id, and prohibited when service_provider = guide_id
            $rules['supplier_id'] = ['required_if:service_provider,supplier_id', 'prohibited_if:service_provider,guide_id', 'nullable', 'integer', 'exists:su_supplier,id'];
            // guide_id is required when service_provider = guide_id, and prohibited when service_provider = supplier_id
            $rules['guide_id'] = ['required_if:service_provider,guide_id', 'prohibited_if:service_provider,supplier_id', 'nullable', 'integer', 'exists:users,id'];
        } else {
            // If financial is nullable, both are optional but still respect service_provider
            $rules['supplier_id'] = ['sometimes', 'prohibited_if:service_provider,guide_id', 'nullable', 'integer', 'exists:su_supplier,id'];
            $rules['guide_id'] = ['sometimes', 'prohibited_if:service_provider,supplier_id', 'nullable', 'integer', 'exists:users,id'];
        }

        return array_merge($rules, [
            'extra_cost' => ['sometimes', 'nullable', 'numeric', 'min:0'],
            'daily_cost' => ['sometimes', $financial, 'numeric', 'min:0'],
            'tip_amount' => ['sometimes', 'nullable', 'numeric', 'min:0'],
            'currency_id' => [$financial, 'integer', 'exists:currencies,id'],
            'executive_id' => ['sometimes', 'nullable', 'integer', 'exists:users,id'],
            'guide_language_id' => ['nullable', 'integer', 'exists:guide_language,id'],
            'people_count' => [$basic, 'integer', 'min:0'],
            'notes' => ['nullable', 'string', 'max:1000'],
            'profile_id' => ['required', 'integer', 'exists:pr_profile,id'],
            'daily_program_id' => [$basic, 'integer', 'exists:pr_daily_programs,id'],
            // selling fields
            'selling_currency_id' => ['sometimes', $selling, 'integer', 'exists:currencies,id'],
            'selling_tax_rate_id' => ['sometimes', 'nullable', 'array'],
            'selling_tax_rate_id.*' => ['integer', 'exists:taxs_rate,id'],
            'selling_price' => ['sometimes', $selling, 'numeric', 'min:0'],
            'selling_total_tax' => ['prohibited'],

        ]);
    }

    public function getUpdateRules(array $request): array
    {
        $basic = $this->authorizeBasic($request, 'edit');
        $financial = $this->authorizeFinancial($request, 'edit');
        $selling = $this->authorizeSelling($request, 'edit');

        $rules = [
            'total_tips' => 'prohibited',
            'grand_total' => 'prohibited',
            'currency_rate' => 'prohibited',
            'is_by_handling' => 'nullabel|boolean',
        ];

        if (array_key_exists('service_id', $request)) {
            $rules['service_id'] = [$basic, 'integer', 'exists:services,id'];
        }
        if (array_key_exists('service_provider', $request)) {
            $rules['service_provider'] = ['required', 'in:supplier_id,guide_id'];
            // Conditional validation based on service_provider
            if ($financial === 'required') {
                // supplier_id is required when service_provider = supplier_id, and prohibited when service_provider = guide_id
                $rules['supplier_id'] = ['required_if:service_provider,supplier_id', 'prohibited_if:service_provider,guide_id', 'nullable', 'integer', 'exists:su_supplier,id'];
                // guide_id is required when service_provider = guide_id, and prohibited when service_provider = supplier_id
                $rules['guide_id'] = ['required_if:service_provider,guide_id', 'prohibited_if:service_provider,supplier_id', 'nullable', 'integer', 'exists:users,id'];
            } else {
                // If financial is nullable, both are optional but still respect service_provider
                $rules['supplier_id'] = ['prohibited_if:service_provider,guide_id', 'nullable', 'integer', 'exists:su_supplier,id'];
                $rules['guide_id'] = ['prohibited_if:service_provider,supplier_id', 'nullable', 'integer', 'exists:users,id'];
            }
        } else {
            // If service_provider is not provided, validate based on what's present
            if (array_key_exists('supplier_id', $request)) {
                $rules['supplier_id'] = [$financial, 'integer', 'exists:su_supplier,id'];
            }
            if (array_key_exists('guide_id', $request)) {
                $rules['guide_id'] = [$financial, 'integer', 'exists:users,id'];
            }
        }
        if (array_key_exists('guide_language_id', $request)) {
            $rules['guide_language_id'] = [$basic, 'integer', 'exists:languages,id'];
        }
        if (array_key_exists('people_count', $request)) {
            $rules['people_count'] = [$basic, 'integer', 'min:0'];
        }
        if (array_key_exists('daily_cost', $request)) {
            $rules['daily_cost'] = [$financial, 'numeric', 'min:0'];
        }
        if (array_key_exists('tip_amount', $request)) {
            $rules['tip_amount'] = ['nullable', 'numeric', 'min:0'];
        }
        if (array_key_exists('extra_cost', $request)) {
            $rules['extra_cost'] = ['nullable', 'numeric', 'min:0'];
        }
        if (array_key_exists('executive_id', $request)) {
            $rules['executive_id'] = ['nullable', 'integer', 'exists:users,id'];
        }
        if (array_key_exists('currency_id', $request)) {
            $rules['currency_id'] = [$financial, 'integer', 'exists:currencies,id'];
        }
        if (array_key_exists('notes', $request)) {
            $rules['notes'] = ['nullable', 'string', 'max:1000'];
        }
        if (array_key_exists('profile_id', $request)) {
            $rules['profile_id'] = ['required', 'integer', 'exists:pr_profile,id'];
        }
        if (array_key_exists('daily_program_id', $request)) {
            $rules['daily_program_id'] = [$basic, 'integer', 'exists:pr_daily_programs,id'];
        }

        if (array_key_exists('is_by_handling', $request)) {
            $rules['is_by_handling'] = ['nullable', 'boolean'];
        }
        if (array_key_exists('selling_currency_id', $request)) {
            $rules['selling_currency_id'] = [$selling, 'integer', 'exists:currencies,id'];
        }
        if (array_key_exists('selling_tax_rate_id', $request)) {
            $rules['selling_tax_rate_id'] = ['nullable', 'array'];
            $rules['selling_tax_rate_id.*'] = ['integer', 'exists:taxs_rate,id'];
        }
        if (array_key_exists('selling_price', $request)) {
            $rules['selling_price'] = [$selling, 'numeric', 'min:0'];
        }
        if (array_key_exists('selling_total_tax', $request)) {
            $rules['selling_total_tax'] = ['prohibited'];
        }

        return $rules;
    }

    public function prepareForCreate(array $request): array|false
    {
        $this->authorizeBasic($request, 'add');
        $this->authorizeFinancial($request, 'add');
        $canFinancial = Gate::allows('ADD_TOUR_REPO_FINANCIAL_INFO');

        // Normalize provider: if service_provider specified, nullify the other id
        $provider = $request['service_provider'] ?? null;
        if ($provider == 'supplier_id') {
            $request['guide_id'] = null;
        } elseif ($provider == 'guide_id') {
            $request['supplier_id'] = null;
            // keep supplier required; do not nullify supplier per business rules
        }
        // Validate service type
        $service = GeneralService::find($request['service_id']);
        if (! $service || $service->type !== ServicesType::TOUR_REPS->value) {
            return ['errors' => ['service_id' => ['service type must be tour_reps']]];
        }
        $request['is_by_handling'] = (bool) ($request['is_by_handling'] ?? false);

        // Set next_pay_date from daily_program day_date (before financial check)
        if (isset($request['daily_program_id'])) {
            $dailyProgram = DailyPrograms::find($request['daily_program_id']);
            if ($dailyProgram && $dailyProgram->day_date) {
                $request['next_pay_date'] = $dailyProgram->day_date;
            }
        }

        if (! $canFinancial) {
            $request['currency_rate'] = 1;
            $request['total_tips'] = 0;
            $request['grand_total'] = 0;

            return $request;
        }

        // Set currency rate
        $currency = isset($request['currency_id']) ? Currency::find($request['currency_id']) : null;
        $request['currency_rate'] = $currency?->exchange_rate ?? 1;

        // Compute totals
        $peopleCount = (int) ($request['people_count'] ?? 0);
        $dailyCost = (float) ($request['daily_cost'] ?? 0);
        $tipAmount = (float) ($request['tip_amount'] ?? 0);
        $extraCost = (float) ($request['extra_cost'] ?? 0);

        $totalTips = $tipAmount * $peopleCount;
        $grandTotal = $dailyCost + $totalTips + $extraCost;

        $request['total_tips'] = $totalTips;
        $request['grand_total'] = $grandTotal;

        // Handle selling (optional)
        $canSelling = Gate::allows('ADD_TOUR_REPO_SELLING_PRICE');
        $profile = isset($request['profile_id']) ? Profile::find($request['profile_id']) : null;
        $eligibleForSelling = $request['is_by_handling'] === true
            || ($profile && (bool) ($profile->has_invoice ?? false));

        if (! $canSelling || ! $eligibleForSelling) {
            $request['selling_total_tax'] = 0;
        } else {
            $sellingCurrencyId = (int) ($request['selling_currency_id'] ?? $request['currency_id'] ?? 0);
            $sellingPrice = (float) ($request['selling_price'] ?? 0);
            $sellingTaxTotal = 0.0;
            $sellingTaxRateIds = $request['selling_tax_rate_id'] ?? [];
            if (is_array($sellingTaxRateIds) && ! empty($sellingTaxRateIds) && $sellingPrice > 0) {
                $rates = TaxRate::whereIn('id', $sellingTaxRateIds)->pluck('percentage');
                $sumPercent = (float) $rates->sum();
                $sellingTaxTotal = ($sellingPrice * $sumPercent) / 100.0;
            }
            $request['selling_total_tax'] = $sellingTaxTotal;
            // Get selling currency rate
            $sellingCurrency = $sellingCurrencyId ? Currency::find($sellingCurrencyId) : null;
            $request['selling_rate'] = $sellingCurrency?->exchange_rate ?? 1;
            $request['selling_currency_id'] = $sellingCurrencyId ?: null;
            $request['selling_price'] = $sellingPrice;
        }

        return $request;
    }

    public function prepareForUpdate(array $request, object $existing): array|false
    {
        $this->authorizeBasic($request, 'edit');
        $this->authorizeFinancial($request, 'edit');
        $canFinancial = Gate::allows('EDIT_TOUR_REPO_FINANCIAL_INFO');

        // Normalize provider switching on update
        if (array_key_exists('service_provider', $request)) {
            if ($request['service_provider'] === 'supplier_id') {
                $request['guide_id'] = null;
            }
        } else {
            if (array_key_exists('supplier_id', $request) && ! is_null($request['supplier_id'])) {
                $request['guide_id'] = null;
            }
        }
        // If service_id provided, ensure it's correct type
        if (array_key_exists('service_id', $request)) {
            $service = GeneralService::find($request['service_id']);
            if (! $service || $service->type !== ServicesType::TOUR_REPS->value) {
                return ['errors' => ['service_id' => ['service type must be tour_reps']]];
            }
        }
        $request['is_by_handling'] = (bool) ($request['is_by_handling'] ?? false);

        if (! $canFinancial) {
            return $request;
        }

        // Refresh currency rate if currency_id provided
        $currencyId = (int) ($request['currency_id'] ?? $existing->currency_id);
        if ($currencyId) {
            $currency = Currency::find($currencyId);
            $request['currency_rate'] = $currency?->exchange_rate ?? ($existing->currency_rate ?? 1);
        }

        // Use provided values or fallback to existing for calculations
        $peopleCount = (int) ($request['people_count'] ?? $existing->people_count ?? 0);
        $dailyCost = (float) ($request['daily_cost'] ?? $existing->daily_cost ?? 0);
        $tipAmount = (float) ($request['tip_amount'] ?? $existing->tip_amount ?? 0);
        $extraCost = (float) ($request['extra_cost'] ?? $existing->extra_cost ?? 0);

        $totalTips = $tipAmount * $peopleCount;
        $grandTotal = $dailyCost + $totalTips + $extraCost;

        $request['total_tips'] = $totalTips;
        $request['grand_total'] = $grandTotal;

        // Handle selling (optional)
        $canSelling = Gate::allows('EDIT_TOUR_REPO_SELLING_PRICE');
        $profileId = (int) ($request['profile_id'] ?? $existing->profile_id ?? 0);
        $profile = $profileId ? Profile::find($profileId) : null;
        $eligibleForSelling = $request['is_by_handling'] === true
            || ($profile && (bool) ($profile->has_invoice ?? false));

        if ($canSelling && $eligibleForSelling) {
            $sellingCurrencyId = (int) ($request['selling_currency_id'] ?? $existing->selling_currency_id ?? $existing->currency_id ?? 0);
            $sellingPrice = (float) ($request['selling_price'] ?? $existing->selling_price ?? 0);
            $sellingTaxRateIds = $request['selling_tax_rate_id'] ?? null;
            if (is_array($sellingTaxRateIds) && $sellingPrice > 0) {
                $rates = TaxRate::whereIn('id', $sellingTaxRateIds)->pluck('percentage');
                $sumPercent = (float) $rates->sum();
                $sellingTotalTax = ($sellingPrice * $sumPercent) / 100.0;
            } else {
                $sellingTotalTax = (float) ($existing->selling_total_tax ?? 0);
            }
            // Get selling currency rate
            $sellingCurrency = $sellingCurrencyId ? Currency::find($sellingCurrencyId) : null;
            $request['selling_rate'] = $sellingCurrency?->exchange_rate ?? ($existing->selling_rate ?? 1);
            $request['selling_currency_id'] = $sellingCurrencyId ?: null;
            $request['selling_price'] = $sellingPrice;
            $request['selling_total_tax'] = $sellingTotalTax;
        }

        return $request;
    }

    public function afterCreate(object $invoiceService, array $request): void
    {
        // No tax persistence for tour_reps
        $this->saveSellingTaxes($invoiceService, $request);
    }

    public function afterUpdate(object $invoiceService, array $request): void
    {
        // No tax persistence for tour_reps
        if (array_key_exists('selling_tax_rate_id', $request)) {
            $this->saveSellingTaxes($invoiceService, $request, true);
        }
    }

    /**
     * Authorize adding/editing basic fields (service_id, guide_language_id, people_count)
     */
    private function authorizeBasic(array $request, string $mode): string
    {
        $ability = $mode === 'edit' ? 'EDIT_TOUR_REPO_BASIC_INFO' : 'ADD_TOUR_REPO_BASIC_INFO';

        return Gate::allows($ability) ? 'required' : 'nullable';
    }

    /**
     * Authorize adding/editing financial fields (provider/supplier/guide/costs/tips/currency/extra)
     */
    private function authorizeFinancial(array $request, string $mode): string
    {
        $ability = $mode === 'edit' ? 'EDIT_TOUR_REPO_FINANCIAL_INFO' : 'ADD_TOUR_REPO_FINANCIAL_INFO';

        return Gate::allows($ability) ? 'required' : 'nullable';
    }

    /**
     * Authorize adding/editing selling fields
     */
    private function authorizeSelling(array $request, string $mode): string
    {
        $ability = $mode === 'edit' ? 'EDIT_TOUR_REPO_SELLING_PRICE' : 'ADD_TOUR_REPO_SELLING_PRICE';

        if (! Gate::allows($ability)) {
            return 'nullable';
        }

        // في create: required فقط لو profile->has_invoice == 1
        if ($mode === 'add') {
            $profileId = (int) ($request['profile_id'] ?? 0);
            if ($profileId > 0) {
                $profile = Profile::find($profileId);
                if ($profile && (bool) ($profile->has_invoice ?? false)) {
                    return 'required';
                }
            }

            return 'nullable';
        }

        // في update: required فقط لو is_by_handling == 1
        if ($mode === 'edit') {
            $isByHandling = (bool) ($request['is_by_handling'] ?? false);

            return $isByHandling ? 'required' : 'nullable';
        }

        return 'nullable';
    }

    private function saveSellingTaxes(object $invoiceService, array $request, bool $replace = false): void
    {
        $taxRateIds = $request['selling_tax_rate_id'] ?? [];
        if (! is_array($taxRateIds) || empty($taxRateIds)) {
            if ($replace) {
                InvoiceServicesSellingTax::where('invoice_services_id', $invoiceService->id)->delete();
            }

            return;
        }

        if ($replace) {
            InvoiceServicesSellingTax::where('invoice_services_id', $invoiceService->id)->delete();
        }

        $sellingPrice = (float) ($request['selling_price'] ?? $invoiceService->selling_price ?? 0);
        if ($sellingPrice <= 0) {
            return;
        }

        $profileId = (int) ($invoiceService->profile_id ?? null);
        $currencyId = (int) ($request['selling_currency_id'] ?? $invoiceService->selling_currency_id ?? $invoiceService->currency_id ?? null);

        $rates = TaxRate::whereIn('id', $taxRateIds)->get(['id', 'percentage']);
        foreach ($rates as $rate) {
            $amount = ($sellingPrice * (float) $rate->percentage) / 100.0;
            InvoiceServicesSellingTax::create([
                'invoice_services_id' => $invoiceService->id,
                'tax_rate_id' => $rate->id,
                'tax_amount' => $amount,
                'tax_rate_amount' => (float) $rate->percentage,
                'currency_id' => $currencyId ?: null,
                'profile_id' => $profileId ?: null,
            ]);
        }
    }
}
